package secrets

import (
	"context"
	"encoding/base64"
	"encoding/json"
	"fmt"
	"net/http"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/secretsmanager"
	"github.com/aws/aws-sdk-go/service/secretsmanager/secretsmanageriface"
)

type Secrets interface {
	GetSecretValue(ctx context.Context, secretID string) (string, error)
	GetTahoeReplicaSecret(ctx context.Context, secretID string) (TahoeSecret, error)
}

// Client is a wrapper for the dynamodb api
type Client struct {
	env            string
	secretsmanager secretsmanageriface.SecretsManagerAPI
}

func NewClient(env string, region string) (*Client, error) {
	sess, err := session.NewSession(&aws.Config{
		HTTPClient: &http.Client{
			Timeout: 3 * time.Second,
			Transport: &http.Transport{
				MaxIdleConnsPerHost: 200,
			},
		},
		MaxRetries: aws.Int(3),
		Region:     aws.String(region),
	})
	if err != nil {
		return nil, fmt.Errorf("secretsmanager: failed to initialize secrets manager client: %w", err)
	}

	return &Client{
		env:            env,
		secretsmanager: secretsmanager.New(sess),
	}, nil
}

func (c *Client) GetSecretValue(ctx context.Context, secretID string) (string, error) {
	result, err := c.secretsmanager.GetSecretValueWithContext(ctx, &secretsmanager.GetSecretValueInput{
		SecretId: aws.String(secretID),
	})
	if err != nil {
		return "", fmt.Errorf("secretsmanager: getSecretValue: %w", err)
	}
	if result == nil {
		return "", fmt.Errorf("secretsmanager: secret value is nil")
	}

	// Copied from sample code in the AWS console UI
	// Depending on whether the secret is a string or binary, one of these fields will be populated.
	if result.SecretString != nil {
		return *result.SecretString, nil
	}

	decodedBinarySecretBytes := make([]byte, base64.StdEncoding.DecodedLen(len(result.SecretBinary)))
	length, err := base64.StdEncoding.Decode(decodedBinarySecretBytes, result.SecretBinary)
	if err != nil {
		return "", err
	}

	return string(decodedBinarySecretBytes[:length]), nil
}

type TahoeSecret struct {
	DBClusterIdentifier string `json:"dbClusterIdentifier"`
	Username            string `json:"username"`
	Password            string `json:"password"`
	DBName              string `json:"dbname"`
	Engine              string `json:"engine"`
	Port                int    `json:"port"`
	Host                string `json:"host"`
}

func (c *Client) GetTahoeReplicaSecret(ctx context.Context, secretID string) (TahoeSecret, error) {
	result, err := c.GetSecretValue(ctx, secretID)
	if err != nil {
		return TahoeSecret{}, err
	}

	var parsedSecret TahoeSecret
	err = json.Unmarshal([]byte(result), &parsedSecret)
	if err != nil {
		return TahoeSecret{}, err
	}

	err = validateTahoeTapSecret(parsedSecret)
	if err != nil {
		return TahoeSecret{}, err
	}

	return parsedSecret, nil
}

func validateTahoeTapSecret(s TahoeSecret) error {
	if s.Host == "" {
		return fmt.Errorf("secret Host is empty")
	}

	if s.Password == "" {
		return fmt.Errorf("secret Password is empty")
	}

	if s.DBName == "" {
		return fmt.Errorf("secret DBName is empty")
	}

	if s.Username == "" {
		return fmt.Errorf("secret Username is empty")
	}

	if s.Port == 0 {
		return fmt.Errorf("secret Port is 0")
	}

	return nil
}
