package dashy

import (
	"context"

	"code.justin.tv/cb/dashy/view/chat"
	"code.justin.tv/cb/dashy/view/clips"
	"code.justin.tv/cb/dashy/view/commercials"
	"code.justin.tv/cb/dashy/view/followers"
	"code.justin.tv/cb/dashy/view/layout"
	"code.justin.tv/cb/dashy/view/multichannel"
	"code.justin.tv/cb/dashy/view/raid"
	"code.justin.tv/cb/dashy/view/stats"
	"code.justin.tv/cb/dashy/view/subscriptionpurchase"
	"code.justin.tv/cb/dashy/view/summary"
	"code.justin.tv/cb/dashy/view/tagfilters"
	"code.justin.tv/cb/dashy/view/tips"
	"code.justin.tv/cb/dashy/view/uniqueviews"
	"code.justin.tv/cb/dashy/view/vpdemographics"
	"code.justin.tv/foundation/twitchclient"
)

// Client is the client interface for the Dashy service.
type Client interface {
	// GetV1LayoutPreferences makes a GET request for a channel's
	// dashboard layout preferences
	GetV1LayoutPreferences(ctx context.Context, params *layout.GetV1LayoutPreferencesParams, reqOpts *twitchclient.ReqOpts) (*layout.Response, error)

	// GetV1MultiChannelChatActivitiesByTime makes a GET request
	// for the chat activities of multiple channels, given a time range.
	GetV1MultiChannelChatActivitiesByTime(ctx context.Context, params *GetV1MultiChannelChatActivitiesByTimeReqParams, reqOpts *twitchclient.ReqOpts) (*multichannel.ChatActivitiesResponse, error)

	// GetV1MultiChannelConcurrentViewersByTime makes a GET request
	// for the concurrent viewer time series data of multiple channels,
	// given a time range.
	GetV1MultiChannelConcurrentViewersByTime(ctx context.Context, params *GetV1MultiChannelConcurrentViewersByTimeReqParams, reqOpts *twitchclient.ReqOpts) (*multichannel.ConcurrentsResponse, error)

	// GetV1MultiChannelCommercialsByTime makes a GET request
	// for the commercial events of multiple channels, given a time range.
	GetV1MultiChannelCommercialsByTime(ctx context.Context, params *GetV1MultiChannelCommercialsByTimeReqParams, reqOpts *twitchclient.ReqOpts) (*multichannel.CommercialsResponse, error)

	// GetV1MultiChannelSessionsByTime makes a GET request
	// for the video play demographics data of multiple channels, given a time range.
	GetV1MultiChannelSessionsByTime(ctx context.Context, params *GetV1MultiChannelSessionsByTimeReqParams, reqOpts *twitchclient.ReqOpts) (*multichannel.SessionsResponse, error)

	// GetV1MultiChannelVideoPlayDemographicsByTime makes a GET request
	// for the video play demographics data of multiple channels, given a time range.
	GetV1MultiChannelVideoPlayDemographicsByTime(ctx context.Context, params *GetV1MultiChannelVideoPlayDemographicsByTimeReqParams, reqOpts *twitchclient.ReqOpts) (*multichannel.DemographicsResponse, error)

	// GetV1MultiChannelFollowsByTime makes a GET request
	// for the concurrent viewer time series data of multiple channels,
	// given a time range.
	GetV1MultiChannelFollowsByTime(ctx context.Context, params *GetV1MultiChannelFollowsByTimeReqParams, reqOpts *twitchclient.ReqOpts) (*multichannel.FollowsResponse, error)

	// GetV1StatsByTime makes a GET request for a channel's
	// statistics within a time range.
	GetV1StatsByTime(ctx context.Context, params *GetV1StatsByTimeReqParams, reqOpts *twitchclient.ReqOpts) (*stats.Response, error)

	// GetV1SessionsSummaryByTime makes a GET request for a list of a
	// channel's complete past sessions, possibly including
	// an ongoing session, within a time range.
	GetV1SessionsSummaryByTime(ctx context.Context, params *GetV1SessionsSummaryByTimeReqParams, reqOpts *twitchclient.ReqOpts) (*summary.Response, error)

	// GetV1FollowersByTime makes a GET request for a channel's
	// followers activity within a time range
	GetV1FollowersByTime(ctx context.Context, params *followers.GetV1FollowersByTimeReqParams, reqOpts *twitchclient.ReqOpts) (*followers.Response, error)

	// GetV1ClipsByTime makes a GET request for a channel's
	// Clips activity within a time range
	GetV1ClipsByTime(ctx context.Context, params *clips.GetV1ClipsByTimeReqParams, reqOpts *twitchclient.ReqOpts) (*clips.Response, error)

	// GetV1VideoPlayDemographicsByTime makes a GET request to the Dashy Service for
	// demographic breakdowns for all video plays
	GetV1VideoPlayDemographicsByTime(ctx context.Context, params *vpdemographics.GetV1VideoPlayDemographicsByTimeReqParams, reqOpts *twitchclient.ReqOpts) (*vpdemographics.Response, error)

	// GetV1ChatActivityByTime makes a GET request for a channel's
	// Chat Activity within a time range
	GetV1ChatActivityByTime(ctx context.Context, params *chat.GetV1ChatActivityByTimeReqParams, reqOpts *twitchclient.ReqOpts) (*chat.Response, error)

	// GetV1UniqueViews makes a GET request for a channel's
	// unique and total views activity within a time range
	GetV1UniqueViews(ctx context.Context, params *uniqueviews.GetV1UniqueViewsByTimeReqParams, reqOpts *twitchclient.ReqOpts) (*uniqueviews.Response, error)

	// GetV1CommercialsByTime makes a GET request to get mid-roll commercials played within a time range
	GetV1CommercialsByTime(ctx context.Context, params *commercials.GetV1CommercialsByTimeReqParams, reqOpts *twitchclient.ReqOpts) (*commercials.Response, error)

	// GetV1Tip gets a random tip in the given language
	GetV1Tip(ctx context.Context, language string, reqOpts *twitchclient.ReqOpts) (*tips.Response, error)

	// GetV1RaidActivityByTime makes a GET request for a channel's
	// Raid Activity within a time range
	GetV1RaidActivityByTime(ctx context.Context, params *raid.GetV1RaidActivityTimeReqParams, reqOpts *twitchclient.ReqOpts) (*raid.Response, error)

	// GetV1SubscriptionPurchaseSuccessByTime makes a GET request for a channel's
	// subscription_purchase_success Activity within a time range
	GetV1SubscriptionPurchaseSuccessByTime(ctx context.Context, params *subscriptionpurchase.GetV1SubscriptionPurchaseActivityTimeReqParams, reqOpts *twitchclient.ReqOpts) (*subscriptionpurchase.Response, error)

	GetV1TagFilters(ctx context.Context, channelID int64, reqOpts *twitchclient.ReqOpts) (*tagfilters.Response, error)
}

type client struct {
	twitchclient.Client
}

// NewClient instantiates a new client for the Dashy service.
func NewClient(config twitchclient.ClientConf) (Client, error) {
	if config.TimingXactName == "" {
		config.TimingXactName = "cb-dashy"
	}

	twitchClient, err := twitchclient.NewClient(config)
	if err != nil {
		return nil, err
	}

	return &client{twitchClient}, nil
}
