package dashy

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"
	"time"

	"code.justin.tv/cb/dashy/view/followers"
	"code.justin.tv/foundation/twitchclient"
	"github.com/stretchr/testify/suite"
)

type GetV1FollowersByTimeSuite struct {
	suite.Suite
}

func TestGetV1FollowersByTimeSuite(t *testing.T) {
	suite.Run(t, &GetV1FollowersByTimeSuite{})
}

func (suite *GetV1FollowersByTimeSuite) TestBadTimeRangeRequest() {
	client, err := NewClient(twitchclient.ClientConf{
		Host: "some-host",
	})
	suite.NoError(err)

	endTime := time.Now()
	twoYearsLaterTime := endTime.AddDate(2, 0, 0)

	params := &followers.GetV1FollowersByTimeReqParams{
		ChannelID: "123",
		StartTime: endTime,
		EndTime:   twoYearsLaterTime,
	}

	_, err = client.GetV1FollowersByTime(context.Background(), params, nil)
	suite.Error(err)
}

func (suite *GetV1FollowersByTimeSuite) TestFailure() {
	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusInternalServerError)
	}))

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	suite.NoError(err)

	startTime := time.Now()
	endTime := startTime.Add(1 * time.Minute)

	params := &followers.GetV1FollowersByTimeReqParams{
		ChannelID: "123",
		StartTime: startTime,
		EndTime:   endTime,
	}

	resp, err := client.GetV1FollowersByTime(context.Background(), params, nil)
	suite.NotNil(err)
	suite.Nil(resp)
}

func (suite *GetV1FollowersByTimeSuite) TestMalformedResponse() {
	payload := `
		{
			"status": 200,
			"message": "malformed JSON String with extra double-quotes""
		}
	`

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)

		_, err := w.Write([]byte(payload))
		suite.NoError(err)
	}))

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	suite.NoError(err)

	startTime := time.Now()
	endTime := startTime.Add(1 * time.Minute)

	params := &followers.GetV1FollowersByTimeReqParams{
		ChannelID: "123",
		StartTime: startTime,
		EndTime:   endTime,
	}

	resp, err := client.GetV1FollowersByTime(context.Background(), params, nil)
	suite.NotNil(err)

	jsonResponse, err := json.Marshal(resp)
	suite.NoError(err)
	suite.NotEqual(string(jsonResponse), payload)
}

func (suite *GetV1FollowersByTimeSuite) TestSuccess() {
	channelID := "123"
	now := time.Now()
	startTime := time.Date(now.Year(), now.Month(), now.Day(), 0, 0, 0, 0, time.UTC)
	endTime := startTime.Add(1 * time.Minute)
	sessionGapDurationMinutes := 15
	interpChunkMinutes := 5
	followerChangeCount := int64(20)
	formattedStartTime := startTime.UTC().Format(time.RFC3339)
	formattedEndTime := endTime.UTC().Format(time.RFC3339)

	payloadFmt := `
		{
			"status": 200,
			"message": "",
			"meta": {
				"start_time": "%s",
				"end_time": "%s",
				"session_gap_duration_minutes": %d,
				"interpolation_chunk_minutes": %d
			},
			"data": {
				"followers_change": [
					{
						"timestamp": "%s",
						"count": %d
					}
				]
			}
		}
	`

	payload := fmt.Sprintf(
		payloadFmt,
		formattedStartTime,
		formattedEndTime,
		sessionGapDurationMinutes,
		interpChunkMinutes,
		formattedStartTime,
		followerChangeCount,
	)

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		suite.Equal(fmt.Sprintf("/v1/stats/channels/%s/followers", channelID), r.URL.EscapedPath())
		suite.Equal(formattedStartTime, r.URL.Query().Get("start_time"))
		suite.Equal(formattedEndTime, r.URL.Query().Get("end_time"))

		w.WriteHeader(http.StatusOK)

		_, err := w.Write([]byte(payload))
		suite.NoError(err)
	}))

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	suite.NoError(err)

	params := &followers.GetV1FollowersByTimeReqParams{
		ChannelID: channelID,
		StartTime: startTime,
		EndTime:   endTime,
	}

	resp, err := client.GetV1FollowersByTime(context.Background(), params, nil)
	suite.NoError(err)

	if suite.NotNil(resp) {
		if suite.NotNil(resp.Meta.StartTime) {
			suite.Equal(startTime.UTC(), resp.Meta.StartTime.UTC())
		}

		if suite.NotNil(resp.Meta.EndTime) {
			suite.Equal(endTime.UTC(), resp.Meta.EndTime.UTC())
		}

		suite.Equal(sessionGapDurationMinutes, resp.Meta.SessionGapDurationMinutes)
		suite.Equal(interpChunkMinutes, resp.Meta.InterpolationChunkMinutes)

		if suite.Len(resp.Data.FollowersChange, 1) {
			change := resp.Data.FollowersChange[0]

			if suite.NotNil(change.Timestamp) {
				suite.Equal(startTime.UTC(), change.Timestamp.UTC())
			}

			suite.Equal(followerChangeCount, change.Count)
		}
	}
}
