package dashy

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/cb/dashy/view/layout"
	"code.justin.tv/foundation/twitchclient"
	"github.com/stretchr/testify/suite"
)

type GetV1LayoutPreferencesSuite struct {
	suite.Suite
}

func TestGetV1LayoutPreferencesSuite(t *testing.T) {
	suite.Run(t, &GetV1LayoutPreferencesSuite{})
}

func (suite *GetV1LayoutPreferencesSuite) TestBadRequest() {
	client, err := NewClient(twitchclient.ClientConf{
		Host: "some-host",
	})
	suite.NoError(err)

	params := &layout.GetV1LayoutPreferencesParams{
		ChannelID: "0",
	}

	_, err = client.GetV1LayoutPreferences(context.Background(), params, nil)
	suite.Error(err)
}

func (suite *GetV1LayoutPreferencesSuite) TestFailure() {
	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusInternalServerError)
	}))

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	suite.NoError(err)

	params := &layout.GetV1LayoutPreferencesParams{
		ChannelID: "123",
	}

	resp, err := client.GetV1LayoutPreferences(context.Background(), params, nil)
	suite.NotNil(err)
	suite.Nil(resp)
}

func (suite *GetV1LayoutPreferencesSuite) TestMalformedResponse() {
	payload := `
		{
			"status": 200,
			"message": "malformed JSON String with extra double-quotes""
		}
	`

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)

		_, err := w.Write([]byte(payload))
		suite.NoError(err)
	}))

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	suite.NoError(err)

	params := &layout.GetV1LayoutPreferencesParams{
		ChannelID: "123",
	}

	resp, err := client.GetV1LayoutPreferences(context.Background(), params, nil)
	suite.NotNil(err)

	jsonResponse, err := json.Marshal(resp)
	suite.NoError(err)
	suite.NotEqual(string(jsonResponse), payload)
}

func (suite *GetV1LayoutPreferencesSuite) TestSuccess() {
	channelID := "123"
	testConfig := "{}"

	payloadFmt := `
		{
			"status": 200,
			"message": "",
			"config": "%s"
		}
	`

	payload := fmt.Sprintf(
		payloadFmt,
		testConfig,
	)

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		suite.Equal(fmt.Sprintf("/v1/dashboard/channels/%s/layout", channelID), r.URL.EscapedPath())

		w.WriteHeader(http.StatusOK)

		_, err := w.Write([]byte(payload))
		suite.NoError(err)
	}))

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	suite.NoError(err)

	params := &layout.GetV1LayoutPreferencesParams{
		ChannelID: channelID,
	}

	resp, err := client.GetV1LayoutPreferences(context.Background(), params, nil)
	suite.NoError(err)

	if suite.NotNil(resp) {
		suite.Equal(resp.Config, testConfig)
	}
}
