package dashy

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"strings"
	"testing"
	"time"

	"code.justin.tv/foundation/twitchclient"
	"github.com/stretchr/testify/suite"
)

type GetV1MultiChannelCommercialsByTimeSuite struct {
	suite.Suite
}

func TestGetV1MultiChannelCommercialsByTimeSuite(t *testing.T) {
	suite.Run(t, &GetV1MultiChannelCommercialsByTimeSuite{})
}

func (suite *GetV1MultiChannelCommercialsByTimeSuite) TestFailure() {
	channelIDs := []string{"123", "abc"}
	startTime := time.Now()
	endTime := startTime.Add(1 * time.Minute)

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusInternalServerError)
	}))

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})

	suite.Require().NoError(err)

	params := &GetV1MultiChannelCommercialsByTimeReqParams{
		ChannelIDs: channelIDs,
		StartTime:  startTime,
		EndTime:    endTime,
	}

	resp, err := client.GetV1MultiChannelCommercialsByTime(context.Background(), params, nil)
	suite.NotNil(err)
	suite.Nil(resp)
}

func (suite *GetV1MultiChannelCommercialsByTimeSuite) TestMalformedResponse() {
	channelIDs := []string{"123", "abc"}
	startTime := time.Now()
	endTime := startTime.Add(1 * time.Minute)

	payload := `
		{
			"status": 200,
			"message": "malformed JSON string with extra double-quotes""
		}
	`

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)

		_, err := w.Write([]byte(payload))
		suite.NoError(err)
	}))

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})

	suite.Require().NoError(err)

	params := &GetV1MultiChannelCommercialsByTimeReqParams{
		ChannelIDs: channelIDs,
		StartTime:  startTime,
		EndTime:    endTime,
	}

	resp, err := client.GetV1MultiChannelCommercialsByTime(context.Background(), params, nil)
	suite.NotNil(err)

	jsonResponse, err := json.Marshal(resp)
	suite.NoError(err)
	suite.NotEqual(string(jsonResponse), payload)
}

func (suite *GetV1MultiChannelCommercialsByTimeSuite) TestSuccess() {
	channelIDs := []string{"123", "abc"}
	now := time.Now()
	startTime := time.Date(now.Year(), now.Month(), now.Day(), 0, 0, 0, 0, time.UTC)
	endTime := startTime.Add(1 * time.Minute)
	durationSeconds := int64(30)
	formattedStartTime := startTime.UTC().Format(time.RFC3339)
	formattedEndTime := endTime.UTC().Format(time.RFC3339)

	payloadFmt := `
		{
			"status": 200,
			"meta": {
				"start_time": "%s",
				"end_time": "%s"
			},
			"data": {
				"%s": [
					{
						"timestamp": "%s",
						"duration_seconds": %d
					}
				]
			}
		}
	`

	payload := fmt.Sprintf(
		payloadFmt,
		formattedStartTime,
		formattedEndTime,
		channelIDs[0],
		formattedStartTime,
		durationSeconds,
	)

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		suite.Equal("/v1/stats/multi_channel/commercials", r.URL.EscapedPath())
		suite.Equal(strings.Join(channelIDs, ","), r.URL.Query().Get("channel_ids"))
		suite.Equal(formattedStartTime, r.URL.Query().Get("start_time"))
		suite.Equal(formattedEndTime, r.URL.Query().Get("end_time"))

		w.WriteHeader(http.StatusOK)

		_, err := w.Write([]byte(payload))
		suite.NoError(err)
	}))

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})

	suite.Require().NoError(err)

	params := &GetV1MultiChannelCommercialsByTimeReqParams{
		ChannelIDs: channelIDs,
		StartTime:  startTime,
		EndTime:    endTime,
	}

	resp, err := client.GetV1MultiChannelCommercialsByTime(context.Background(), params, nil)
	suite.NoError(err)

	if suite.NotNil(resp) {
		if suite.NotNil(resp.Meta.StartTime) {
			suite.Equal(startTime, *resp.Meta.StartTime)
		}

		if suite.NotNil(resp.Meta.EndTime) {
			suite.Equal(endTime, *resp.Meta.EndTime)
		}

		if suite.Contains(resp.Data, channelIDs[0]) {
			commercials := resp.Data[channelIDs[0]]

			if suite.Len(commercials, 1) {
				if suite.NotNil(commercials[0].Timestamp) {
					suite.Equal(startTime, *commercials[0].Timestamp)
				}

				suite.Equal(durationSeconds, commercials[0].DurationSeconds)
			}
		}
	}
}
