package dashy

import (
	"context"
	"net/http"
	"strings"
	"time"

	"code.justin.tv/cb/dashy/view/multichannel"
	"code.justin.tv/foundation/twitchclient"
)

// GetV1MultiChannelSessionsByTimeReqParams contains
// the required parameters to make the
// GET /v1/stats/multi_channel/sessions request.
type GetV1MultiChannelSessionsByTimeReqParams struct {
	ChannelIDs []string
	StartTime  time.Time
	EndTime    time.Time
}

// GetV1MultiChannelSessionsByTime makes a GET request
// to the Dashy service for stream sessions during the given time range.
func (c *client) GetV1MultiChannelSessionsByTime(ctx context.Context, params *GetV1MultiChannelSessionsByTimeReqParams, reqOpts *twitchclient.ReqOpts) (*multichannel.SessionsResponse, error) {
	path := "/v1/stats/multi_channel/sessions"

	req, err := c.NewRequest(http.MethodGet, path, nil)
	if err != nil {
		return nil, err
	}

	req.Header.Set("Content-Type", "application/json")

	query := req.URL.Query()
	query.Set("channel_ids", strings.Join(params.ChannelIDs, ","))
	query.Set("start_time", params.StartTime.UTC().Format(time.RFC3339))
	query.Set("end_time", params.EndTime.UTC().Format(time.RFC3339))
	req.URL.RawQuery = query.Encode()

	mergedOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.cb-dashy.get_multi_channel_sessions",
		StatSampleRate: 0.1,
	})

	response := &multichannel.SessionsResponse{}
	_, err = c.DoJSON(ctx, response, req, mergedOpts)
	if err != nil {
		return nil, err
	}

	return response, nil
}
