package dashy

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"strings"
	"testing"
	"time"

	"code.justin.tv/cb/dashy/view/multichannel"
	"code.justin.tv/foundation/twitchclient"
	"github.com/stretchr/testify/suite"
)

type GetV1MultiChannelSessionsByTimeSuite struct {
	suite.Suite
}

func TestGetV1MultiChannelSessionsByTimeSuite(t *testing.T) {
	suite.Run(t, &GetV1MultiChannelSessionsByTimeSuite{})
}

func (suite *GetV1MultiChannelSessionsByTimeSuite) TestFailure() {
	channelIDs := []string{"123", "abc"}
	startTime := time.Now()
	endTime := startTime.Add(1 * time.Minute)

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusInternalServerError)
	}))

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})

	suite.Require().NoError(err)

	params := &GetV1MultiChannelSessionsByTimeReqParams{
		ChannelIDs: channelIDs,
		StartTime:  startTime,
		EndTime:    endTime,
	}

	resp, err := client.GetV1MultiChannelSessionsByTime(context.Background(), params, nil)
	suite.NotNil(err)
	suite.Nil(resp)
}

func (suite *GetV1MultiChannelSessionsByTimeSuite) TestMalformedResponse() {
	channelIDs := []string{"123", "abc"}
	startTime := time.Now()
	endTime := startTime.Add(1 * time.Minute)

	payload := `
		{
			"status": 200,
			"message": "malformed JSON string with extra double-quotes""
		}
	`

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)

		_, err := w.Write([]byte(payload))
		suite.NoError(err)
	}))

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})

	suite.Require().NoError(err)

	params := &GetV1MultiChannelSessionsByTimeReqParams{
		ChannelIDs: channelIDs,
		StartTime:  startTime,
		EndTime:    endTime,
	}

	resp, err := client.GetV1MultiChannelSessionsByTime(context.Background(), params, nil)
	suite.NotNil(err)

	jsonResponse, err := json.Marshal(resp)
	suite.NoError(err)
	suite.NotEqual(string(jsonResponse), payload)
}

func (suite *GetV1MultiChannelSessionsByTimeSuite) TestSuccess() {
	channelIDs := []string{"123", "abc"}
	now := time.Now()
	startTime := time.Date(now.Year(), now.Month(), now.Day(), 0, 0, 0, 0, time.UTC)
	endTime := startTime.Add(1 * time.Minute)
	formattedStartTime := startTime.UTC().Format(time.RFC3339)
	formattedEndTime := endTime.UTC().Format(time.RFC3339)

	payloadFmt := `
		{
			"status": 200,
			"meta": {
				"start_time": "%s",
				"end_time": "%s"
			},
			"data": {
				"%s": [
					{
						"start_time": "%s",
						"end_time": "%s"
					}
				]
			}
		}
	`

	payload := fmt.Sprintf(
		payloadFmt,
		formattedStartTime,
		formattedEndTime,
		channelIDs[0],
		formattedStartTime,
		formattedEndTime,
	)

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		suite.Equal("/v1/stats/multi_channel/sessions", r.URL.EscapedPath())
		suite.Equal(strings.Join(channelIDs, ","), r.URL.Query().Get("channel_ids"))
		suite.Equal(formattedStartTime, r.URL.Query().Get("start_time"))
		suite.Equal(formattedEndTime, r.URL.Query().Get("end_time"))

		w.WriteHeader(http.StatusOK)

		_, err := w.Write([]byte(payload))
		suite.NoError(err)
	}))

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})

	suite.Require().NoError(err)

	params := &GetV1MultiChannelSessionsByTimeReqParams{
		ChannelIDs: channelIDs,
		StartTime:  startTime,
		EndTime:    endTime,
	}

	resp, err := client.GetV1MultiChannelSessionsByTime(context.Background(), params, nil)
	suite.NoError(err)

	if suite.NotNil(resp) {
		expected := map[string][]multichannel.Session{
			channelIDs[0]: {
				{
					StartTime: &startTime,
					EndTime:   &endTime,
				},
			},
		}

		suite.Equal(expected, resp.Data)
	}
}
