package dashy

import (
	"context"
	"errors"
	"net/http"
	"strings"
	"time"

	"code.justin.tv/cb/dashy/view/multichannel"
	"code.justin.tv/foundation/twitchclient"
)

// GetV1MultiChannelVideoPlayDemographicsByTimeReqParams contains
// the required parameters to make the
// GET /v1/stats/multi_channel/video_play_demographics request.
type GetV1MultiChannelVideoPlayDemographicsByTimeReqParams struct {
	ChannelIDs []string
	StartTime  time.Time
	EndTime    time.Time
}

// Validate ensures that GetV1MultiChannelVideoPlayDemographicsByTimeReqParams
// is valid for Client.GetV1MultiChannelVideoPlayDemographicsByTime.
func (p *GetV1MultiChannelVideoPlayDemographicsByTimeReqParams) Validate() error {
	if len(p.ChannelIDs) == 0 {
		return errors.New("dashy: channel ids cannot be empty")
	}

	if p.StartTime.After(p.EndTime) {
		return errors.New("dashy: start time cannot be after end time")
	}

	return nil
}

// GetV1MultiChannelVideoPlayDemographicsByTime makes a GET request
// to the Dashy service for the concurrent viewer time series data
// of one or more channels, given a time range.
func (c *client) GetV1MultiChannelVideoPlayDemographicsByTime(ctx context.Context, params *GetV1MultiChannelVideoPlayDemographicsByTimeReqParams, reqOpts *twitchclient.ReqOpts) (*multichannel.DemographicsResponse, error) {
	path := "/v1/stats/multi_channel/video_play_demographics"

	req, err := c.NewRequest(http.MethodGet, path, nil)
	if err != nil {
		return nil, err
	}

	req.Header.Set("Content-Type", "application/json")

	query := req.URL.Query()
	query.Set("channel_ids", strings.Join(params.ChannelIDs, ","))
	query.Set("start_time", params.StartTime.UTC().Format(time.RFC3339))
	query.Set("end_time", params.EndTime.UTC().Format(time.RFC3339))
	req.URL.RawQuery = query.Encode()

	mergedOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.cb-dashy.get_multi_channel_video_play_demographics",
		StatSampleRate: 0.1,
	})

	response := &multichannel.DemographicsResponse{}
	_, err = c.DoJSON(ctx, response, req, mergedOpts)
	if err != nil {
		return nil, err
	}

	return response, nil
}
