package dashy

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"strings"
	"testing"
	"time"

	"code.justin.tv/cb/dashy/view/multichannel"
	"code.justin.tv/foundation/twitchclient"
	"github.com/stretchr/testify/suite"
)

type GetV1MultiChannelVideoPlayDemographicsByTimeSuite struct {
	suite.Suite
}

func TestGetV1MultiChannelVideoPlayDemographicsByTimeSuite(t *testing.T) {
	suite.Run(t, &GetV1MultiChannelVideoPlayDemographicsByTimeSuite{})
}

func (suite *GetV1MultiChannelVideoPlayDemographicsByTimeSuite) TestInvalidChannelIDs() {
	startTime := time.Now()
	twoYearsLaterTime := startTime.AddDate(2, 0, 0)

	params := GetV1MultiChannelVideoPlayDemographicsByTimeReqParams{
		StartTime: startTime,
		EndTime:   twoYearsLaterTime,
	}

	suite.Error(params.Validate())
}

func (suite *GetV1MultiChannelVideoPlayDemographicsByTimeSuite) TestInvalidTimeRange() {
	channelIDs := []string{"123", "abc"}
	endTime := time.Now()
	twoYearsLaterTime := endTime.AddDate(2, 0, 0)

	params := GetV1MultiChannelVideoPlayDemographicsByTimeReqParams{
		ChannelIDs: channelIDs,
		StartTime:  twoYearsLaterTime,
		EndTime:    endTime,
	}

	suite.Error(params.Validate())
}

func (suite *GetV1MultiChannelVideoPlayDemographicsByTimeSuite) TestFailure() {
	channelIDs := []string{"123", "abc"}
	startTime := time.Now()
	endTime := startTime.Add(1 * time.Minute)

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusInternalServerError)
	}))

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})

	suite.Require().NoError(err)

	params := &GetV1MultiChannelVideoPlayDemographicsByTimeReqParams{
		ChannelIDs: channelIDs,
		StartTime:  startTime,
		EndTime:    endTime,
	}

	resp, err := client.GetV1MultiChannelVideoPlayDemographicsByTime(context.Background(), params, nil)
	suite.NotNil(err)
	suite.Nil(resp)
}

func (suite *GetV1MultiChannelVideoPlayDemographicsByTimeSuite) TestMalformedResponse() {
	channelIDs := []string{"123", "abc"}
	startTime := time.Now()
	endTime := startTime.Add(1 * time.Minute)

	payload := `
		{
			"status": 200,
			"message": "malformed JSON string with extra double-quotes""
		}
	`

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)

		_, err := w.Write([]byte(payload))
		suite.NoError(err)
	}))

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})

	suite.Require().NoError(err)

	params := &GetV1MultiChannelVideoPlayDemographicsByTimeReqParams{
		ChannelIDs: channelIDs,
		StartTime:  startTime,
		EndTime:    endTime,
	}

	resp, err := client.GetV1MultiChannelVideoPlayDemographicsByTime(context.Background(), params, nil)
	suite.NotNil(err)

	jsonResponse, err := json.Marshal(resp)
	suite.NoError(err)
	suite.NotEqual(string(jsonResponse), payload)
}

func (suite *GetV1MultiChannelVideoPlayDemographicsByTimeSuite) TestSuccess() {
	channelIDs := []string{"123"}
	now := time.Now()
	startTime := time.Date(now.Year(), now.Month(), now.Day(), 0, 0, 0, 0, time.UTC)
	endTime := startTime.Add(1 * time.Minute)
	formattedStartTime := startTime.UTC().Format(time.RFC3339)
	formattedEndTime := endTime.UTC().Format(time.RFC3339)
	videoPlayCount := int64(2775)
	geo := "US"
	geoCount := int64(1000)
	platform := "web"
	platformCount := int64(777)
	internal := "hosted"
	internalCount := int64(123)
	external := "google"
	externalCount := int64(3)

	payloadFmt := `
		{
			"status": 200,
			"meta": {
				"start_time": "%s",
				"end_time": "%s"
			},
			"data": {
				"video_play_count": %d,
				"geo": {
					"%s": %d
				},
				"platform": {
					"%s": %d
				},
				"referrer": {
					"internal": {
						"%s": %d
					},
					"external": {
						"%s": %d
					}
				}
			}
		}
	`

	payload := fmt.Sprintf(
		payloadFmt,
		formattedStartTime,
		formattedEndTime,
		videoPlayCount,
		geo,
		geoCount,
		platform,
		platformCount,
		internal,
		internalCount,
		external,
		externalCount,
	)

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		suite.Equal("/v1/stats/multi_channel/video_play_demographics", r.URL.EscapedPath())
		suite.Equal(strings.Join(channelIDs, ","), r.URL.Query().Get("channel_ids"))
		suite.Equal(formattedStartTime, r.URL.Query().Get("start_time"))
		suite.Equal(formattedEndTime, r.URL.Query().Get("end_time"))

		w.WriteHeader(http.StatusOK)

		_, err := w.Write([]byte(payload))
		suite.NoError(err)
	}))

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})

	suite.Require().NoError(err)

	params := &GetV1MultiChannelVideoPlayDemographicsByTimeReqParams{
		ChannelIDs: channelIDs,
		StartTime:  startTime,
		EndTime:    endTime,
	}

	resp, err := client.GetV1MultiChannelVideoPlayDemographicsByTime(context.Background(), params, nil)
	suite.NoError(err)

	if suite.NotNil(resp) {
		expected := multichannel.Demographics{
			VideoPlayCount: videoPlayCount,
			Geo: map[string]int64{
				geo: geoCount,
			},
			Platform: map[string]int64{
				platform: platformCount,
			},
			Referrer: multichannel.Referrer{
				Internal: map[string]int64{
					internal: internalCount,
				},
				External: map[string]int64{
					external: externalCount,
				},
			},
		}

		suite.Exactly(expected, resp.Data)
	}
}
