package dashy

import (
	"context"
	"errors"
	"fmt"
	"time"

	"code.justin.tv/cb/dashy/view/summary"
	"code.justin.tv/foundation/twitchclient"
)

// GetV1SessionsSummaryByTimeReqParams contains the REQUIRED
// parameters for making a GET request for a channel's
// sessions within StartTime and EndTime.
type GetV1SessionsSummaryByTimeReqParams struct {
	ChannelID int64
	StartTime time.Time
	EndTime   time.Time
}

func (c *client) GetV1SessionsSummaryByTime(ctx context.Context, params *GetV1SessionsSummaryByTimeReqParams, reqOpts *twitchclient.ReqOpts) (*summary.Response, error) {
	if params.EndTime.Before(params.StartTime) {
		msg := "GetV1SessionsSummaryByTimeReqParams.StartTime must be before GetV1SessionsSummaryByTimeReqParams.EndTime"
		return nil, errors.New(msg)
	}

	path := fmt.Sprintf("/v1/stats/channels/%d/sessions_summary", params.ChannelID)

	req, err := c.NewRequest("GET", path, nil)
	if err != nil {
		return nil, err
	}

	req.Header.Set("Content-Type", "application/json")

	query := req.URL.Query()
	query.Add("start_time", params.StartTime.UTC().Format(time.RFC3339))
	query.Add("end_time", params.EndTime.UTC().Format(time.RFC3339))
	req.URL.RawQuery = query.Encode()

	mergedOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.cb-dashy.get_sessions_summary_by_time",
		StatSampleRate: 0.1,
	})

	response := &summary.Response{}
	_, err = c.DoJSON(ctx, response, req, mergedOpts)
	if err != nil {
		return nil, err
	}

	return response, nil
}
