package dashy

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"
	"time"

	"code.justin.tv/cb/dashy/view/summary"
	"code.justin.tv/foundation/twitchclient"
	"github.com/stretchr/testify/suite"
)

type GetV1SessionsSummaryByTimeSuite struct {
	suite.Suite
}

func TestGetV1SessionsSummaryByTimeSuite(t *testing.T) {
	suite.Run(t, &GetV1SessionsSummaryByTimeSuite{})
}

func (suite *GetV1SessionsSummaryByTimeSuite) TestBadTimeRangeRequest() {
	client, err := NewClient(twitchclient.ClientConf{
		Host: "some-host",
	})
	suite.NoError(err)

	endTime := time.Now()
	twoYearsLaterTime := endTime.AddDate(2, 0, 0)

	params := &GetV1SessionsSummaryByTimeReqParams{
		ChannelID: 123,
		StartTime: endTime,
		EndTime:   twoYearsLaterTime,
	}

	_, err = client.GetV1SessionsSummaryByTime(context.Background(), params, nil)
	suite.Error(err)
}

func (suite *GetV1SessionsSummaryByTimeSuite) TestFailure() {
	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusInternalServerError)
	}))

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	suite.NoError(err)

	startTime := time.Now()
	endTime := startTime.Add(1 * time.Minute)

	params := &GetV1SessionsSummaryByTimeReqParams{
		ChannelID: 123,
		StartTime: startTime,
		EndTime:   endTime,
	}

	resp, err := client.GetV1SessionsSummaryByTime(context.Background(), params, nil)
	suite.Error(err)
	suite.Nil(resp)
}

func (suite *GetV1SessionsSummaryByTimeSuite) TestMalformedResponse() {
	payload := `
		{
			"status": 200,
			"message": "malformed JSON string with extra double-quotes""
		}
	`

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)

		_, err := w.Write([]byte(payload))
		suite.NoError(err)
	}))

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	suite.NoError(err)

	startTime := time.Now()
	endTime := startTime.Add(1 * time.Minute)

	params := &GetV1SessionsSummaryByTimeReqParams{
		ChannelID: 123,
		StartTime: startTime,
		EndTime:   endTime,
	}

	resp, err := client.GetV1SessionsSummaryByTime(context.Background(), params, nil)
	suite.Error(err)
	suite.Nil(resp)
}

func (suite *GetV1SessionsSummaryByTimeSuite) TestSuccess() {
	channelID := int64(123)
	now := time.Now()
	startTime := time.Date(now.Year(), now.Month(), now.Day(), 0, 0, 0, 0, time.UTC)
	endTime := startTime.Add(1 * time.Minute)
	formattedStartTime := startTime.UTC().Format(time.RFC3339)
	formattedEndTime := endTime.UTC().Format(time.RFC3339)
	avgConcurrentViewerCount := "2775.315789473684"
	maxConcurrentViewersCount := "9001"
	minutesWatched := "1234567"
	followersChange := "1234"
	broadcastID := "999999"
	maxViewersTime := startTime
	messagesTotal := "123"
	chattersUnique := "12"
	commercialCount := "10"
	commercialDensity := "0.5"
	commercialLengthTotal := "100"
	raidsIncoming := "23"
	subscriptionsNew := "34"
	videoPlayTotal := "55"
	videoPlayUnique := "44"
	videoPlayExternal := []*summary.Stat{
		{
			Key:   "www.google.com",
			Value: "111",
		},
		{
			Key:   "t.co",
			Value: "222",
		},
	}
	videoPlayExternalJSON, err := json.Marshal(videoPlayExternal)
	videoPlayInternal := []*summary.Stat{
		{
			Key:   "followed_channel",
			Value: "333",
		},
		{
			Key:   "directory_browse",
			Value: "444",
		},
	}
	videoPlayInternalJSON, err := json.Marshal(videoPlayInternal)
	videoPlayGeographical := []*summary.Stat{
		{
			Key:   "US",
			Value: "555",
		},
		{
			Key:   "DE",
			Value: "666",
		},
	}
	videoPlayGeographicalJSON, err := json.Marshal(videoPlayGeographical)
	videoPlayPlatform := []*summary.Stat{
		{
			Key:   "web",
			Value: "777",
		},
		{
			Key:   "ios",
			Value: "888",
		},
	}
	videoPlayPlatformJSON, err := json.Marshal(videoPlayPlatform)
	suite.NoError(err)

	payloadFmt := `
		{
			"status": 200,
			"message": "",
			"meta": {
				"start_time": "%s",
				"end_time": "%s"
			},
			"sessions": [
				{
					"start_time": "%s",
					"end_time": "%s",
					"summary": {
						"concurrents_average": "%s",
						"concurrents_max": "%s",
						"concurrents_peak_time": "%s",
						"minutes_watched_total": "%s",
						"followers_change": "%s",
						"chatters_unique": "%s",
						"messages_total": "%s",
						"commercial_count": "%s",
						"commercial_density": "%s",
						"commercial_length_total": "%s",
						"raids_incoming": "%s",
						"subscriptions_new": "%s",
						"video_play_total": "%s",
						"video_play_unique": "%s"
					},
					"broadcast_ids": [
						"%s"
					],
					"stats": {
						"video_play_external_referrers": %s,
						"video_play_internal_referrers": %s,
						"video_play_geographics": %s,
						"video_play_platforms": %s
					}
				}
			]
		}
	`
	payload := fmt.Sprintf(
		payloadFmt,
		formattedStartTime,
		formattedEndTime,
		formattedStartTime,
		formattedEndTime,
		avgConcurrentViewerCount,
		maxConcurrentViewersCount,
		maxViewersTime.UTC().Format(time.RFC3339),
		minutesWatched,
		followersChange,
		chattersUnique,
		messagesTotal,
		commercialCount,
		commercialDensity,
		commercialLengthTotal,
		raidsIncoming,
		subscriptionsNew,
		videoPlayTotal,
		videoPlayUnique,
		broadcastID,
		videoPlayExternalJSON,
		videoPlayInternalJSON,
		videoPlayGeographicalJSON,
		videoPlayPlatformJSON,
	)

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		suite.Equal(fmt.Sprintf("/v1/stats/channels/%d/sessions_summary", channelID), r.URL.EscapedPath())
		suite.Equal(formattedStartTime, r.URL.Query().Get("start_time"))
		suite.Equal(formattedEndTime, r.URL.Query().Get("end_time"))

		w.WriteHeader(http.StatusOK)

		_, err = w.Write([]byte(payload))
		suite.NoError(err)
	}))

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	suite.NoError(err)

	params := &GetV1SessionsSummaryByTimeReqParams{
		ChannelID: channelID,
		StartTime: startTime,
		EndTime:   endTime,
	}

	resp, err := client.GetV1SessionsSummaryByTime(context.Background(), params, nil)
	suite.NoError(err)

	if suite.NotNil(resp) {
		if suite.NotNil(resp.Meta.StartTime) {
			suite.Equal(startTime.UTC(), resp.Meta.StartTime.UTC())
		}

		if suite.NotNil(resp.Meta.EndTime) {
			suite.Equal(endTime.UTC(), resp.Meta.EndTime.UTC())
		}

		if suite.Len(resp.Sessions, 1) {
			if suite.NotNil(resp.Sessions[0].StartTime) {
				suite.Equal(startTime.UTC(), resp.Sessions[0].StartTime.UTC())
			}

			if suite.NotNil(resp.Sessions[0].EndTime) {
				suite.Equal(endTime.UTC(), resp.Sessions[0].EndTime.UTC())
			}

			suite.Equal(avgConcurrentViewerCount, *resp.Sessions[0].Summary["concurrents_average"])
			suite.Equal(maxConcurrentViewersCount, *resp.Sessions[0].Summary["concurrents_max"])
			suite.Equal(minutesWatched, *resp.Sessions[0].Summary["minutes_watched_total"])
			suite.Equal(followersChange, *resp.Sessions[0].Summary["followers_change"])
			suite.Equal(messagesTotal, *resp.Sessions[0].Summary["messages_total"])
			suite.Equal(chattersUnique, *resp.Sessions[0].Summary["chatters_unique"])
			suite.Equal(commercialCount, *resp.Sessions[0].Summary["commercial_count"])
			suite.Equal(commercialDensity, *resp.Sessions[0].Summary["commercial_density"])
			suite.Equal(commercialLengthTotal, *resp.Sessions[0].Summary["commercial_length_total"])
			suite.Equal(raidsIncoming, *resp.Sessions[0].Summary["raids_incoming"])
			suite.Equal(subscriptionsNew, *resp.Sessions[0].Summary["subscriptions_new"])
			suite.Equal(videoPlayTotal, *resp.Sessions[0].Summary["video_play_total"])
			suite.Equal(videoPlayUnique, *resp.Sessions[0].Summary["video_play_unique"])
			suite.Equal(maxViewersTime.UTC().Format(time.RFC3339), *resp.Sessions[0].Summary["concurrents_peak_time"])
			suite.Equal(broadcastID, resp.Sessions[0].BroadcastIDs[0])
			suite.Equal(videoPlayExternal, resp.Sessions[0].Stats["video_play_external_referrers"])
			suite.Equal(videoPlayInternal, resp.Sessions[0].Stats["video_play_internal_referrers"])
			suite.Equal(videoPlayGeographical, resp.Sessions[0].Stats["video_play_geographics"])
			suite.Equal(videoPlayPlatform, resp.Sessions[0].Stats["video_play_platforms"])
		}
	}
}
