package dashy

import (
	"context"
	"errors"
	"fmt"
	"time"

	"code.justin.tv/cb/dashy/view/stats"
	"code.justin.tv/foundation/twitchclient"
)

// GetV1StatsByTimeReqParams contains the REQUIRED
// parameters for making a GET request for a channel's
// stats within StartTime and EndTime.
type GetV1StatsByTimeReqParams struct {
	ChannelID int64
	StartTime time.Time
	EndTime   time.Time
}

// GetV1StatsByTime makes a GET request to the Dashy service
// for a collection of stats data within a time range.
func (c *client) GetV1StatsByTime(ctx context.Context, params *GetV1StatsByTimeReqParams, reqOpts *twitchclient.ReqOpts) (*stats.Response, error) {
	if params.EndTime.Before(params.StartTime) {
		msg := "GetStatsByTimeReqParams.StartTime must be before GetStatsByTimeReqParams.EndTime"
		return nil, errors.New(msg)
	}

	path := fmt.Sprintf("/v1/stats/channels/%d/stats", params.ChannelID)

	req, err := c.NewRequest("GET", path, nil)
	if err != nil {
		return nil, err
	}

	req.Header.Set("Content-Type", "application/json")

	query := req.URL.Query()
	query.Add("start_time", params.StartTime.UTC().Format(time.RFC3339))
	query.Add("end_time", params.EndTime.UTC().Format(time.RFC3339))
	req.URL.RawQuery = query.Encode()

	mergedOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.cb-dashy.get_stats_by_time",
		StatSampleRate: 0.1,
	})

	response := &stats.Response{}
	_, err = c.DoJSON(ctx, response, req, mergedOpts)
	if err != nil {
		return nil, err
	}

	return response, nil
}
