package dashy

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"
	"time"

	"code.justin.tv/cb/dashy/view/vpdemographics"
	"code.justin.tv/foundation/twitchclient"
	"github.com/stretchr/testify/suite"
)

type GetV1VideoPlayDemographicsByTimeSuite struct {
	suite.Suite
}

func TestGetV1VideoPlayDemographicsByTimeSuite(t *testing.T) {
	suite.Run(t, &GetV1VideoPlayDemographicsByTimeSuite{})
}

func (suite *GetV1VideoPlayDemographicsByTimeSuite) TestBadTimeRangeRequest() {
	client, err := NewClient(twitchclient.ClientConf{
		Host: "some-host",
	})
	suite.NoError(err)

	endTime := time.Now()
	twoYearsLaterTime := endTime.AddDate(2, 0, 0)

	params := &vpdemographics.GetV1VideoPlayDemographicsByTimeReqParams{
		ChannelID: "123",
		StartTime: endTime,
		EndTime:   twoYearsLaterTime,
	}

	_, err = client.GetV1VideoPlayDemographicsByTime(context.Background(), params, nil)
	suite.Error(err)
}

func (suite *GetV1VideoPlayDemographicsByTimeSuite) TestFailure() {
	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusInternalServerError)
	}))

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	suite.NoError(err)

	startTime := time.Now()
	endTime := startTime.Add(1 * time.Minute)

	params := &vpdemographics.GetV1VideoPlayDemographicsByTimeReqParams{
		ChannelID: "123",
		StartTime: startTime,
		EndTime:   endTime,
	}

	resp, err := client.GetV1VideoPlayDemographicsByTime(context.Background(), params, nil)
	suite.NotNil(err)
	suite.Nil(resp)
}

func (suite *GetV1VideoPlayDemographicsByTimeSuite) TestMalformedResponse() {
	payload := `
		{
			"status": 200,
			"message": "malformed JSON String with extra double-quotes""
		}
	`

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)

		_, err := w.Write([]byte(payload))
		suite.NoError(err)
	}))

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	suite.NoError(err)

	startTime := time.Now()
	endTime := startTime.Add(1 * time.Minute)

	params := &vpdemographics.GetV1VideoPlayDemographicsByTimeReqParams{
		ChannelID: "123",
		StartTime: startTime,
		EndTime:   endTime,
	}

	resp, err := client.GetV1VideoPlayDemographicsByTime(context.Background(), params, nil)
	suite.NotNil(err)

	jsonResponse, err := json.Marshal(resp)
	suite.NoError(err)
	suite.NotEqual(string(jsonResponse), payload)
}

func (suite *GetV1VideoPlayDemographicsByTimeSuite) TestSuccess() {
	channelID := "123"
	now := time.Now()
	startTime := time.Date(now.Year(), now.Month(), now.Day(), 0, 0, 0, 0, time.UTC)
	endTime := startTime.Add(1 * time.Minute)

	sessionGapDurationMinutes := 15

	countryA := "Country A"
	countryB := "Country B"

	platformA := "Platform A"
	platformB := "Platform B"

	locationA := "Location A"
	locationB := "Location B"

	testCountA := 420
	testCountB := 1337

	formattedStartTime := startTime.UTC().Format(time.RFC3339)
	formattedEndTime := endTime.UTC().Format(time.RFC3339)

	payloadFmt := `
		{
			"status": 200,
			"message": "",
			"meta": {
				"start_time": "%s",
				"end_time": "%s",
				"session_gap_duration_minutes": %d
			},
			"data": {
				"demographics": [
					{
		                "timestamp": "%s",
		                "geo": {
		                    "%s": %d
		                },
		                "platform": {
		                    "%s": %d,
		                    "%s": %d
		                },
		                "referral": {
		                    "internal": {
		                    	"%s": %d
		                    },
		                    "external": {
		                    	"%s": %d
		                    }
		                }
		            },
		            {
		                "timestamp": "%s",
		                "geo": {
		                    "%s": %d,
		                    "%s": %d
		                },
		                "platform": {
		                    "%s": %d
		                },
		                "referral": {
		                    "internal": null,
		                    "external": null
		                }
		            }
				]
			}
		}
	`

	payload := fmt.Sprintf(
		payloadFmt,
		formattedStartTime,
		formattedEndTime,
		sessionGapDurationMinutes,
		formattedStartTime,
		countryA,
		testCountA,
		platformA,
		testCountA,
		platformB,
		testCountB,
		locationA,
		testCountA,
		locationB,
		testCountB,
		formattedEndTime,
		countryA,
		testCountA,
		countryB,
		testCountB,
		platformB,
		testCountB,
	)

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		suite.Equal(fmt.Sprintf("/v1/stats/channels/%s/video_play_demographics", channelID), r.URL.EscapedPath())
		suite.Equal(formattedStartTime, r.URL.Query().Get("start_time"))
		suite.Equal(formattedEndTime, r.URL.Query().Get("end_time"))

		w.WriteHeader(http.StatusOK)

		_, err := w.Write([]byte(payload))
		suite.NoError(err)
	}))

	defer server.Close()

	client, err := NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	suite.NoError(err)

	params := &vpdemographics.GetV1VideoPlayDemographicsByTimeReqParams{
		ChannelID: channelID,
		StartTime: startTime,
		EndTime:   endTime,
	}

	resp, err := client.GetV1VideoPlayDemographicsByTime(context.Background(), params, nil)
	suite.NoError(err)

	if suite.NotNil(resp) {
		if suite.NotNil(resp.Meta.StartTime) {
			suite.Equal(startTime.UTC(), resp.Meta.StartTime.UTC())
		}

		if suite.NotNil(resp.Meta.EndTime) {
			suite.Equal(endTime.UTC(), resp.Meta.EndTime.UTC())
		}

		suite.Equal(sessionGapDurationMinutes, resp.Meta.SessionGapDurationMinutes)

		if suite.Len(resp.Data.Demographics, 2) {
			datumA := resp.Data.Demographics[0]
			datumB := resp.Data.Demographics[1]

			if suite.NotNil(datumA.Timestamp) {
				suite.Equal(startTime.UTC(), datumA.Timestamp.UTC())
			}
			if suite.NotNil(datumB.Timestamp) {
				suite.Equal(endTime.UTC(), datumB.Timestamp.UTC())
			}

			if suite.Len(datumA.Geographical, 1) &&
				suite.Contains(datumA.Geographical, countryA) {
				suite.Equal(datumA.Geographical[countryA], int64(testCountA))
			}
			if suite.Len(datumB.Geographical, 2) {
				if suite.Contains(datumB.Geographical, countryA) {
					suite.Equal(datumB.Geographical[countryA], int64(testCountA))
				}

				if suite.Contains(datumB.Geographical, countryB) {
					suite.Equal(datumB.Geographical[countryB], int64(testCountB))
				}
			}

			if suite.Len(datumA.Platform, 2) {
				if suite.Contains(datumA.Platform, platformA) {
					suite.Equal(datumA.Platform[platformA], int64(testCountA))
				}

				if suite.Contains(datumA.Platform, platformB) {
					suite.Equal(datumA.Platform[platformB], int64(testCountB))
				}
			}
			if suite.Len(datumB.Platform, 1) &&
				suite.Contains(datumB.Platform, platformB) {
				suite.Equal(datumB.Platform[platformB], int64(testCountB))
			}

			if suite.Len(datumA.Referral.Internal, 1) && suite.Contains(datumA.Referral.Internal, locationA) {
				suite.Equal(datumA.Referral.Internal[locationA], int64(testCountA))
			}
			if suite.Len(datumA.Referral.External, 1) && suite.Contains(datumA.Referral.External, locationB) {
				suite.Equal(datumA.Referral.External[locationB], int64(testCountB))
			}

			suite.Len(datumB.Referral.Internal, 0)
			suite.Len(datumB.Referral.External, 0)
		}
	}
}
