package main

import (
	"context"
	"net/http"
	"os"
	"os/signal"
	"syscall"
	"time"

	"code.justin.tv/cb/dashy/config"
	"code.justin.tv/cb/dashy/internal/api"
	"code.justin.tv/cb/dashy/internal/auth"
	"code.justin.tv/cb/dashy/internal/clients"
	_ "github.com/lib/pq"
	log "github.com/sirupsen/logrus"
)

const (
	readTimeout     = 5 * time.Second
	writeTimeout    = 10 * time.Second
	shutdownTimeout = 15 * time.Second
)

func init() {
	config.SetupRollbarLogging()
	config.Load()
}

func main() {
	env := config.Environment

	decoder, err := auth.NewDecoder(config.Values.ECCPublicKeyPath)
	if err != nil {
		log.WithError(err).Fatal("Failed to instantiate authorization decoder")
		return
	}

	clientsWrapper, err := clients.NewClients()
	if err != nil {
		log.WithError(err).Fatal("Failed to instantiate clients")
		return
	}

	log.Info("Web application starting in ", env)

	server := &http.Server{
		Addr: ":8000",
		Handler: api.NewServer(&api.ServerParams{
			AuthDecoder: decoder,
			Clients:     clientsWrapper,
			StatsD:      &clientsWrapper.Statter,
		}),
		ReadTimeout:  readTimeout,
		WriteTimeout: writeTimeout,
	}

	go func() {
		log.Info("Server listening on http://localhost", server.Addr)
		if err := server.ListenAndServe(); err != http.ErrServerClosed {
			log.WithError(err).Fatal("Server failed fatally while listening")
		}
	}()

	gracefulShutdown(server)
}

func gracefulShutdown(server *http.Server) {
	stop := make(chan os.Signal, 1)
	signal.Notify(stop, os.Interrupt, syscall.SIGTERM)

	<-stop

	log.Infof("Shutting down server with %s timeout", shutdownTimeout)

	ctx, cancel := context.WithTimeout(context.Background(), shutdownTimeout)
	defer cancel()

	if err := server.Shutdown(ctx); err != nil {
		log.WithError(err).Fatal("Server failed to shut down")
		return
	}

	log.Info("Gracefully shut down server")
}
