package api

import (
	"context"
	"net/http"
	"time"

	"code.justin.tv/cb/dashy/internal/httputil"
	"code.justin.tv/cb/dashy/view/chat"
)

const (
	chatStatsChunkSize = 5 // minutes
)

// v1ChatActivity returns the last broadcast session's stats
// of a given channel ID within a specified time range.
func (s *Server) v1ChatActivity(w http.ResponseWriter, req *http.Request) {
	writer := httputil.NewJSONResponseWriter(w)
	channelID := req.Context().Value(contextKeyChannelID).(int64)

	reqTimeRange := req.Context().Value(contextKeyTimeRange).(timeRange)
	startTime, endTime := reqTimeRange.startTime, reqTimeRange.endTime

	chatTimeSeries, chunkSize, err := s.getChatActivity(req.Context(), channelID, startTime, endTime)
	if err != nil {
		writer.InternalServerError("Failed to get chat activity counts", err)
		return
	}

	response := &chat.Response{
		Status: http.StatusOK,
		Meta: chat.Meta{
			StartTime:                 &startTime,
			EndTime:                   &endTime,
			SessionGapDurationMinutes: sessionMinuteGap,
			InterpolationChunkMinutes: chunkSize,
		},
		Data: chat.Data{
			Chat: chatTimeSeries,
		},
	}

	writer.OK(response)
}

func (s *Server) getChatActivity(ctx context.Context, channelID int64, startTime time.Time, endTime time.Time) ([]*chat.ChatBreakdown, int, error) {
	chatActivityByTime, err := s.Clients.Zephyr.GetUniqueChattersByTime(ctx, channelID, startTime, endTime)
	if err != nil {
		return nil, 0, err
	}

	// build the output chatTimeSeries by iterating chatActivityByTime
	// and converting the backend data into output data.
	chatTimeSeries := make([]*chat.ChatBreakdown, len(chatActivityByTime))
	for idx, chatActivity := range chatActivityByTime {
		timestamp := chatActivity.Time

		chatTimeSeries[idx] = &chat.ChatBreakdown{
			Timestamp:    &timestamp,
			Participants: chatActivity.UniqueChatters,
			Messages:     chatActivity.TotalMessages,
		}
	}

	return chatTimeSeries, chatStatsChunkSize, err
}
