package api

import (
	"context"
	"net/http"
	"time"

	"code.justin.tv/cb/dashy/internal/httputil"
	"code.justin.tv/cb/dashy/view/clips"

	log "github.com/sirupsen/logrus"
)

// Clip represents a clip
type Clip struct {
	ChannelID      int64
	Time           time.Time
	ClipID         string
	Creates        int64
	Plays          int64
	PlaysBreakdown map[string]int64
}

// v1Clips returns the last broadcast session's Clips stats
// of a given channel ID within a specified time range.
func (s *Server) v1Clips(w http.ResponseWriter, req *http.Request) {
	writer := httputil.NewJSONResponseWriter(w)
	channelID := req.Context().Value(contextKeyChannelID).(int64)

	reqTimeRange := req.Context().Value(contextKeyTimeRange).(timeRange)
	startTime, endTime := reqTimeRange.startTime, reqTimeRange.endTime

	fetchedClips, err := s.getClips(req.Context(), channelID, startTime, endTime)
	if err != nil {
		writer.InternalServerError("Failed to get clips data", err)
		return
	}

	clipsData := make([]*clips.Clips, len(fetchedClips))
	for idx := 0; idx < len(fetchedClips); idx++ {
		clipsData[idx] = &clips.Clips{
			Timestamp:      &fetchedClips[idx].Time,
			Plays:          fetchedClips[idx].Plays,
			Creates:        fetchedClips[idx].Creates,
			PlaysBreakdown: fetchedClips[idx].PlaysBreakdown,
		}
	}

	response := &clips.Response{
		Status: http.StatusOK,
		Meta: clips.Meta{
			StartTime:                 &startTime,
			EndTime:                   &endTime,
			SessionGapDurationMinutes: sessionMinuteGap,
		},
		Data: clips.Data{
			Clips: clipsData,
		},
	}

	writer.OK(response)

}

func (s *Server) getClips(ctx context.Context, channelID int64, startTime, endTime time.Time) ([]Clip, error) {
	clipCreates, err := s.Zephyr.GetClipCreatesByTime(ctx, channelID, startTime, endTime)
	if err != nil {
		return []Clip{}, err
	}

	clipPlays, err := s.Zephyr.GetClipPlaysByTime(ctx, channelID, startTime, endTime)
	if err != nil {
		return []Clip{}, err
	}

	// We should never have plays for clips that don't have a creation entry. If we find too many
	// clip plays here, we have bad data in dynamo and we should log it.
	if len(clipPlays) > len(clipCreates) {
		log.WithFields(log.Fields{
			"channel_id": channelID,
			"start_time": startTime,
			"end_time":   endTime,
		}).Warn("Zephyr clips: Found clip plays in dynamo without corresponding clip create timestamps")
	}

	// The length of clipCreates should always be greater or equal the length of clipPlays. We iterate
	// through the clipCreates array to ensure we get all the creates, and that interval
	// should always cover all clip plays.
	clipModels := make([]Clip, len(clipCreates))
	clipPlayIdx := 0
	for clipCreateIdx := 0; clipCreateIdx < len(clipCreates); clipCreateIdx++ {
		clip := Clip{
			ChannelID:      clipCreates[clipCreateIdx].ChannelID,
			Time:           clipCreates[clipCreateIdx].Time,
			Creates:        clipCreates[clipCreateIdx].Creates,
			PlaysBreakdown: map[string]int64{}, // make an empty map here so we don't return any nil values
		}

		// Clip plays should always match the clip create timestamp. Here we check if the
		// timestamps are the same, and if they are, we update our clip model with the plays data.
		// If they aren't, we ignore plays and continue with the same clipPlayIdx until we find a
		// matching time.
		if clipPlayIdx < len(clipPlays) {
			if clipCreates[clipCreateIdx].Time.Equal(clipPlays[clipPlayIdx].Time) {
				clip.Plays = clipPlays[clipPlayIdx].Plays
				clip.PlaysBreakdown = clipPlays[clipPlayIdx].Referrers

				clipPlayIdx++
			}
		}

		clipModels[clipCreateIdx] = clip
	}

	return clipModels, nil
}
