package api

import (
	"context"
	"net/http"
	"time"

	"code.justin.tv/cb/dashy/internal/clients/zephyr"
	"code.justin.tv/cb/dashy/internal/httputil"
	"code.justin.tv/cb/dashy/view/commercials"
)

const (
	SingleAdLength = 30
)

// Commercial represents a commercial
type Commercial struct {
	ChannelID int64
	Length    int64
	Time      zephyr.DynamoTimestamp
}

func (s *Server) v1Commercials(w http.ResponseWriter, req *http.Request) {
	writer := httputil.NewJSONResponseWriter(w)
	channelID := req.Context().Value(contextKeyChannelID).(int64)

	reqTimeRange := req.Context().Value(contextKeyTimeRange).(timeRange)
	startTime, endTime := reqTimeRange.startTime, reqTimeRange.endTime

	commercialsByTime, err := s.getCommercials(req.Context(), channelID, startTime, endTime)
	if err != nil {
		writer.InternalServerError("Failed to get commercials", err)
		return
	}

	commercialsData := make([]*commercials.Commercial, len(commercialsByTime))
	var totalAdLen int64

	for i, comm := range commercialsByTime {
		timestamp := comm.Time.Converted
		totalAdLen += comm.Length

		commercialsData[i] = &commercials.Commercial{
			Timestamp: &timestamp,
			Length:    comm.Length,
		}
	}

	adDensity := GetCommercialDensity(totalAdLen, startTime, endTime)

	response := &commercials.Response{
		Status: http.StatusOK,
		Meta: commercials.Meta{
			StartTime:                 &startTime,
			EndTime:                   &endTime,
			SessionGapDurationMinutes: sessionMinuteGap,
		},
		Data: commercials.Data{
			Commercials: commercialsData,
			Density:     adDensity,
		},
	}

	writer.OK(response)
}

func (s *Server) getCommercials(ctx context.Context, channelID int64, startTime time.Time, endTime time.Time) ([]Commercial, error) {
	commercials, err := s.Zephyr.GetCommercialByTime(ctx, channelID, startTime, endTime)

	if err != nil {
		return nil, err
	}

	commercialTimeSeries := make([]Commercial, len(commercials))
	for idx, commercial := range commercials {

		commercialTimeSeries[idx] = Commercial{
			Time: zephyr.DynamoTimestamp{
				Converted: commercial.Time,
			},
			Length:    commercial.CommercialLength,
			ChannelID: commercial.ChannelID,
		}
	}

	return commercialTimeSeries, nil
}

// We calculate ad density based on ads per hour. However,
// we can use the session duration in minutes in order to
// get an ad density number for session durations under one hour
func GetCommercialDensity(totalAdLength int64, startTime time.Time, endTime time.Time) float64 {
	var commDensity float64
	num30sAds := float64(totalAdLength / SingleAdLength)
	sessionLen := endTime.Sub(startTime).Hours()

	if sessionLen > 0 {
		commDensity = num30sAds / sessionLen
	}

	return commDensity
}
