package api

import (
	"context"
	"math"
	"net/http"
	"time"

	"code.justin.tv/cb/dashy/internal/httputil"
	"code.justin.tv/cb/dashy/view/followers"
)

const (
	followersStatsChunkSize             = 5 // minutes
	followersInterpolationPointEstimate = 85
)

type rawFollowersChange []*followers.FollowersChangeCount

// v1Followers returns the last broadcast session's stats
// of a given channel ID within a specified time range.
func (s *Server) v1Followers(w http.ResponseWriter, req *http.Request) {
	writer := httputil.NewJSONResponseWriter(w)
	channelID := req.Context().Value(contextKeyChannelID).(int64)

	reqTimeRange := req.Context().Value(contextKeyTimeRange).(timeRange)
	startTime, endTime := reqTimeRange.startTime, reqTimeRange.endTime

	followersChange, chunkSize, err := s.getFollowers(req.Context(), channelID, startTime, endTime)

	if err != nil {
		writer.InternalServerError("Failed to get followers", err)
		return
	}

	response := &followers.Response{
		Status: http.StatusOK,
		Meta: followers.Meta{
			StartTime:                 &startTime,
			EndTime:                   &endTime,
			SessionGapDurationMinutes: sessionMinuteGap,
			InterpolationChunkMinutes: chunkSize,
		},
		Data: followers.Data{
			FollowersChange: followersChange,
		},
	}

	writer.OK(response)
}

func (s *Server) getFollowers(ctx context.Context, channelID int64, startTime time.Time, endTime time.Time) ([]*followers.FollowersChangeCount, int, error) {
	followersAdded, err := s.Clients.Zephyr.GetFollowersByTime(ctx, channelID, startTime, endTime)
	if err != nil {
		return nil, 0, err
	}

	followersTimeSeries := make(rawFollowersChange, len(followersAdded))
	for idx, increment := range followersAdded {
		timestamp := increment.Time
		followersTimeSeries[idx] = &followers.FollowersChangeCount{
			Timestamp: &timestamp,
			Count:     increment.Count,
		}
	}
	interpolated, chunkSize := followersTimeSeries.interpolate(followersInterpolationPointEstimate)

	return interpolated, chunkSize, nil
}

// take a data response object and reduce the number of datapoints such
// that it is less than {points}
func (data rawFollowersChange) interpolate(points int) ([]*followers.FollowersChangeCount, int) {
	// if there arent many points to begin with we dont need to
	//	interpolate
	pointCount := len(data)
	if pointCount < points {
		return data, followersStatsChunkSize
	}

	interpolated := []*followers.FollowersChangeCount{}
	dataRange := data[pointCount-1].Timestamp.Sub(*data[0].Timestamp)
	segmentDuration := time.Duration(followersStatsChunkSize) * time.Minute

	// the number of chunks we should combine to reduce the total number of points
	//	around the number the user requested
	chunkCount := math.Ceil(dataRange.Seconds() / segmentDuration.Seconds() / float64(points))
	// the size of each chunk in minutes
	chunkDurationInt := int(chunkCount * followersStatsChunkSize)
	chunkDuration := time.Duration(chunkDurationInt) * time.Minute

	chunkCutoff := data[0].Timestamp.Add(chunkDuration)
	chunkCurrent := &followers.FollowersChangeCount{
		Count:     data[0].Count,
		Timestamp: data[0].Timestamp,
	}
	// number of points accumulated in each chunk
	numChunkPoints := 1
	idx := 1

	for idx < pointCount {
		if !data[idx].Timestamp.Before(chunkCutoff) {
			if numChunkPoints > 0 {
				interpolated = append(interpolated, chunkCurrent)
			}

			chunkTimestamp := chunkCutoff
			chunkCurrent = &followers.FollowersChangeCount{
				Count:     0,
				Timestamp: &chunkTimestamp,
			}
			numChunkPoints = 0
			chunkCutoff = chunkCutoff.Add(chunkDuration)

			continue
		}

		numChunkPoints++
		// for followers we want the total of followers in the new sample size
		chunkCurrent.Count += data[idx].Count
		idx++
	}

	if numChunkPoints > 0 {
		interpolated = append(interpolated, chunkCurrent)
	}

	return interpolated, chunkDurationInt
}
