package api

import (
	"context"
	"fmt"
	"net/http"
	"strconv"
	"strings"
	"time"

	"code.justin.tv/cb/dashy/internal/clients/zephyr"
	"code.justin.tv/cb/dashy/internal/httputil"
	"code.justin.tv/cb/dashy/internal/set"
	"code.justin.tv/common/goauthorization"
	"goji.io/pat"
)

type contextKey int

const (
	contextKeyChannelID contextKey = iota
	contextKeyChannelIDs
	contextKeyTimeRange
	contextKeyTwitchAuthorizationToken
)

const maxChannelIDCount = 100

type timeRange struct {
	startTime time.Time
	endTime   time.Time
}

func validateChannelID(inner http.Handler) http.Handler {
	middleware := func(w http.ResponseWriter, req *http.Request) {
		writer := httputil.NewJSONResponseWriter(w)

		channelID, err := strconv.ParseInt(pat.Param(req, "channel_id"), 10, 64)
		if err != nil {
			writer.BadRequest("Channel ID must be an integer.")
			return
		}

		ctx := context.WithValue(req.Context(), contextKeyChannelID, channelID)

		inner.ServeHTTP(w, req.WithContext(ctx))
	}

	return http.HandlerFunc(middleware)
}

func validateTimeRange(inner http.HandlerFunc) http.Handler {
	middleware := func(w http.ResponseWriter, req *http.Request) {
		writer := httputil.NewJSONResponseWriter(w)

		startTime, err := time.Parse(time.RFC3339, req.URL.Query().Get("start_time"))
		if err != nil {
			writer.BadRequest("Query parameter 'start_time' must be in RFC 3339 format.")
			return
		}

		endTime, err := time.Parse(time.RFC3339, req.URL.Query().Get("end_time"))
		if err != nil {
			writer.BadRequest("Query parameter 'end_time' must be in RFC 3339 format.")
			return
		}

		if startTime.After(endTime) {
			writer.BadRequest("Query parameter 'start_time' must be before parameter 'end_time.'")
			return
		}

		if startTime.Before(zephyr.Start) {
			// In either of the following cases, the queried time range will be potentially too large,
			// resulting in either a client-side timeout reached,
			// or a data set too large to be written in the HTTP response.
			//
			// - when the current time and therefore `endTime` become much greater than `zephyr.Start`
			// - when more data is backfilled (i.e. `zephyr.Start` updated to be further back in time)
			//
			// This solution does NOT scale!
			startTime = zephyr.Start

			if endTime.Before(startTime) {
				endTime = startTime
			}
		}

		ctx := context.WithValue(req.Context(), contextKeyTimeRange, timeRange{
			startTime: startTime,
			endTime:   endTime,
		})

		inner(w, req.WithContext(ctx))
	}

	return http.HandlerFunc(middleware)
}

func validateQueryParamChannelIDs(inner http.Handler) http.Handler {
	middleware := func(w http.ResponseWriter, req *http.Request) {
		writer := httputil.NewJSONResponseWriter(w)
		queryString := req.URL.Query().Get("channel_ids")

		if queryString == "" {
			writer.BadRequest("Query parameter 'channel_ids' is required")
			return
		}

		channelIDs := strings.Split(queryString, ",")
		channelIDSet := set.NewInt64Set()

		for _, channelID := range channelIDs {
			channelIDInt, err := strconv.ParseInt(channelID, 10, 64)
			if err != nil {
				writer.BadRequest("All channel IDs must be integers")
				return
			}

			channelIDSet.Add(channelIDInt)
		}

		uniques := channelIDSet.Items()

		if len(uniques) > maxChannelIDCount {
			writer.BadRequest(fmt.Sprint("Maximum count of channel IDs is ", maxChannelIDCount))
			return
		}

		ctx := context.WithValue(req.Context(), contextKeyChannelIDs, uniques)

		inner.ServeHTTP(w, req.WithContext(ctx))
	}

	return http.HandlerFunc(middleware)
}

// The following auth middlewares attempt to parse the "Twitch-Authorization"
// JSON web token (JWT) set in the request header.

func (s *Server) authorizeChannelViewStats(inner http.Handler) http.Handler {
	middleware := func(w http.ResponseWriter, req *http.Request) {
		writer := httputil.NewJSONResponseWriter(w)

		token, err := s.authDecoder.ParseToken(req)
		if err != nil {
			writer.Forbidden("Unauthorized to view channel stats: invalid or missing Twitch-Authorization token")
			return
		}

		channelID := pat.Param(req, "channel_id")
		err = s.authDecoder.CanViewStats(token, channelID)
		if err != nil {
			// if claims fail, try getting info from users service
			inErr := s.validateViewStatsWithoutClaims(req.Context(), channelID, token)
			if inErr != nil {
				writer.Forbidden("Unauthorized to view channel stats")
				return
			}
		}

		inner.ServeHTTP(w, withAuthorizationTokenContext(req, token))
	}

	return http.HandlerFunc(middleware)
}

func (s *Server) validateViewStatsWithoutClaims(ctx context.Context, channelID string, token *goauthorization.AuthorizationToken) error {
	userID := token.GetSubject()
	if userID == channelID {
		return nil
	}

	user, err := s.Users.GetUserByID(ctx, userID)
	if user == nil || err != nil {
		return fmt.Errorf("could not get user information for user \"%s\": %v", userID, err)
	}
	if (user.Admin != nil && *user.Admin) || (user.Subadmin != nil && *user.Subadmin) {
		return nil
	}

	return fmt.Errorf("user \"%s\" is not authorized to view stats for channel \"%s\"", userID, channelID)
}

func (s *Server) authorizeMultiChannelViewStats(inner http.Handler) http.Handler {
	middleware := func(w http.ResponseWriter, req *http.Request) {
		writer := httputil.NewJSONResponseWriter(w)

		token, err := s.authDecoder.ParseToken(req)
		if err != nil {
			msg := "Unauthorized to view multi-channel stats: invalid or missing Twitch-Authorization token"
			writer.Forbidden(msg)
			return
		}

		claim, err := token.GetTokenClaims("view_multi_channel_stats")
		if err != nil {
			msg := "Unauthorized to view multi-channel stats: missing capability claim 'view_multi_channel_stats'"
			writer.Forbidden(msg)
			return
		}

		channelIDsCapabilityValue, ok := claim["channel_ids"]
		if !ok {
			writer.Forbidden("Unauthorized to view multi-channel stats: missing claim argument 'channel_ids'")
			return
		}

		commaSeparatedIDs, ok := channelIDsCapabilityValue.(string)
		if !ok {
			writer.Forbidden("Unauthorized to view multi-channel stats: invalid claim value for 'channel_ids'")
			return
		}

		requestedChannelIDs, _ := req.Context().Value(contextKeyChannelIDs).([]int64)
		authorizedChannelIDs := strings.Split(commaSeparatedIDs, ",")
		authorizedChannelIDSet := map[string]struct{}{}

		for _, authorizedChannelID := range authorizedChannelIDs {
			authorizedChannelIDSet[authorizedChannelID] = struct{}{}
		}

		for _, requestedChannelID := range requestedChannelIDs {
			if _, ok := authorizedChannelIDSet[strconv.FormatInt(requestedChannelID, 10)]; !ok {
				writer.Forbidden("Unauthorized to view multi-channel stats")
				return
			}
		}

		inner.ServeHTTP(w, withAuthorizationTokenContext(req, token))
	}

	return http.HandlerFunc(middleware)
}

func withAuthorizationTokenContext(req *http.Request, token *goauthorization.AuthorizationToken) *http.Request {
	return req.WithContext(context.WithValue(req.Context(), contextKeyTwitchAuthorizationToken, token))
}
