package api

import (
	"net/http"
	"strconv"
	"sync"

	"code.justin.tv/cb/dashy/internal/clients/zephyr"
	"code.justin.tv/cb/dashy/internal/httputil"
	"code.justin.tv/cb/dashy/view/multichannel"
	"golang.org/x/sync/errgroup"
)

// GET /v1/stats/multi_channel/chat_activities?channel_ids=&start_time=&end_time=
func (s *Server) v1MultiChannelChatActivities(w http.ResponseWriter, req *http.Request) {
	writer := httputil.NewJSONResponseWriter(w)
	channelIDs := req.Context().Value(contextKeyChannelIDs).([]int64)
	reqTimeRange := req.Context().Value(contextKeyTimeRange).(timeRange)
	startTime, endTime := reqTimeRange.startTime, reqTimeRange.endTime

	group, ctx := errgroup.WithContext(req.Context())
	mutex := sync.Mutex{}
	allChannelChat := map[string][]multichannel.ChatActivity{}

	for _, channelID := range channelIDs {
		channelID := channelID

		group.Go(func() error {
			dbChatActivities, err := s.Clients.Zephyr.GetUniqueChattersByTime(ctx, channelID, startTime, endTime)
			if err != nil {
				return err
			}

			mutex.Lock()
			allChannelChat[strconv.FormatInt(channelID, 10)] = multiChannelChatActivity(dbChatActivities)
			mutex.Unlock()

			return nil
		})
	}

	if err := group.Wait(); err != nil {
		writer.InternalServerError("Failed to query chat activity stats for one or more channels", err)
		return
	}

	response := multichannel.ChatActivitiesResponse{
		Status: http.StatusOK,
		Meta: multichannel.ChatActivitiesMeta{
			StartTime: &startTime,
			EndTime:   &endTime,
		},
		Data: allChannelChat,
	}

	writer.OK(response)
}

func multiChannelChatActivity(chatters []zephyr.UniqueChatter) []multichannel.ChatActivity {
	activities := make([]multichannel.ChatActivity, len(chatters))

	for i := range chatters {
		activities[i] = multichannel.ChatActivity{
			Timestamp:        &chatters[i].Time,
			ParticipantCount: chatters[i].UniqueChatters,
			MessageCount:     chatters[i].TotalMessages,
		}
	}

	return activities
}
