package api

import (
	"net/http"
	"strconv"
	"sync"

	"code.justin.tv/cb/dashy/internal/httputil"
	"code.justin.tv/cb/dashy/view/multichannel"

	"golang.org/x/sync/errgroup"
)

// GET /v1/stats/multi_channel/commercials?channel_ids=&start_time=&end_time=
func (s *Server) v1MultiChannelCommercials(w http.ResponseWriter, req *http.Request) {
	writer := httputil.NewJSONResponseWriter(w)
	channelIDs := req.Context().Value(contextKeyChannelIDs).([]int64)
	reqTimeRange := req.Context().Value(contextKeyTimeRange).(timeRange)
	startTime, endTime := reqTimeRange.startTime, reqTimeRange.endTime

	group, ctx := errgroup.WithContext(req.Context())
	mutex := sync.Mutex{}
	allChannelCommercials := map[string][]multichannel.CommercialEvent{}

	for _, channelID := range channelIDs {
		channelID := channelID

		group.Go(func() error {
			commercials, err := s.getCommercials(ctx, channelID, startTime, endTime)
			if err != nil {
				return err
			}

			mutex.Lock()
			allChannelCommercials[strconv.FormatInt(channelID, 10)] = multiChannelCommercialEvent(commercials)
			mutex.Unlock()

			return nil
		})
	}

	if err := group.Wait(); err != nil {
		writer.InternalServerError("Failed to query commercial stats for one or more channels", err)
		return
	}

	response := multichannel.CommercialsResponse{
		Status: http.StatusOK,
		Meta: multichannel.CommercialsMeta{
			StartTime: &startTime,
			EndTime:   &endTime,
		},
		Data: allChannelCommercials,
	}

	writer.OK(response)
}

func multiChannelCommercialEvent(commercials []Commercial) []multichannel.CommercialEvent {
	events := make([]multichannel.CommercialEvent, len(commercials))

	for i := range commercials {
		events[i] = multichannel.CommercialEvent{
			Timestamp:       &commercials[i].Time.Converted,
			DurationSeconds: commercials[i].Length,
		}
	}

	return events
}
