package api

import (
	"net/http"
	"strconv"
	"sync"

	"code.justin.tv/cb/dashy/internal/clients/zephyr"
	"code.justin.tv/cb/dashy/internal/httputil"
	"code.justin.tv/cb/dashy/view/multichannel"
	"golang.org/x/sync/errgroup"
)

// GET /v1/stats/multi_channel/concurrent_viewers?channel_ids=
func (s *Server) v1MultiChannelConcurrentViewers(w http.ResponseWriter, req *http.Request) {
	writer := httputil.NewJSONResponseWriter(w)
	channelIDs := req.Context().Value(contextKeyChannelIDs).([]int64)
	reqTimeRange := req.Context().Value(contextKeyTimeRange).(timeRange)
	startTime, endTime := reqTimeRange.startTime, reqTimeRange.endTime

	group, ctx := errgroup.WithContext(req.Context())
	mutex := sync.Mutex{}
	allChannelViewership := map[string][]multichannel.ConcurrentViewership{}

	for _, channelID := range channelIDs {
		channelID := channelID

		group.Go(func() error {
			concurrents, err := s.Zephyr.GetConcurrentViewersByTime(ctx, channelID, startTime, endTime)
			if err != nil {
				return err
			}

			mutex.Lock()
			allChannelViewership[strconv.FormatInt(channelID, 10)] = multiChannelConcurrentViewers(concurrents)
			mutex.Unlock()

			return nil
		})
	}

	if err := group.Wait(); err != nil {
		writer.InternalServerError("Failed to query concurrent viewer stats for one or more channels", err)
		return
	}

	response := multichannel.ConcurrentsResponse{
		Status: http.StatusOK,
		Meta: multichannel.ConcurrentsMeta{
			StartTime: &startTime,
			EndTime:   &endTime,
		},
		Data: allChannelViewership,
	}

	writer.OK(response)
}

func multiChannelConcurrentViewers(concurrents []zephyr.ConcurrentViewership) []multichannel.ConcurrentViewership {
	channelViews := make([]multichannel.ConcurrentViewership, len(concurrents))

	for i := range concurrents {
		channelViews[i] = multichannel.ConcurrentViewership{
			Timestamp:    &concurrents[i].Timestamp,
			AverageCount: concurrents[i].AverageCount,
		}
	}

	return channelViews
}
