package api

import (
	"net/http"
	"strconv"
	"sync"

	"code.justin.tv/cb/dashy/internal/clients/zephyr"
	"code.justin.tv/cb/dashy/internal/httputil"
	"code.justin.tv/cb/dashy/view/multichannel"

	"golang.org/x/sync/errgroup"
)

// GET /v1/stats/multi_channel/follows?channel_ids=&start_time=&end_time=
func (s *Server) v1MultiChannelFollows(w http.ResponseWriter, req *http.Request) {
	writer := httputil.NewJSONResponseWriter(w)
	channelIDs := req.Context().Value(contextKeyChannelIDs).([]int64)
	reqTimeRange := req.Context().Value(contextKeyTimeRange).(timeRange)
	startTime, endTime := reqTimeRange.startTime, reqTimeRange.endTime

	group, ctx := errgroup.WithContext(req.Context())
	mutex := sync.Mutex{}
	allChannelSnapshots := map[string][]multichannel.FollowSnapshot{}

	for _, channelID := range channelIDs {
		channelID := channelID

		group.Go(func() error {
			followers, err := s.Clients.Zephyr.GetFollowersByTime(ctx, channelID, startTime, endTime)
			if err != nil {
				return err
			}

			mutex.Lock()
			allChannelSnapshots[strconv.FormatInt(channelID, 10)] = multiChannelFollowSnapshots(followers)
			mutex.Unlock()

			return nil
		})
	}

	if err := group.Wait(); err != nil {
		writer.InternalServerError("Failed to get follower stats for one or more channels", err)
		return
	}

	response := multichannel.FollowsResponse{
		Status: http.StatusOK,
		Meta: multichannel.FollowsMeta{
			StartTime: &startTime,
			EndTime:   &endTime,
		},
		Data: allChannelSnapshots,
	}

	writer.OK(response)
}

func multiChannelFollowSnapshots(newFollowers []zephyr.NewFollower) []multichannel.FollowSnapshot {
	snapshots := make([]multichannel.FollowSnapshot, len(newFollowers))

	for i := range newFollowers {
		snapshots[i] = multichannel.FollowSnapshot{
			Timestamp: &newFollowers[i].Time,
			Count:     newFollowers[i].Count,
		}
	}

	return snapshots
}
