package api

import (
	"net/http"
	"strconv"
	"sync"

	"code.justin.tv/cb/dashy/internal/clients/zephyr"
	"code.justin.tv/cb/dashy/internal/httputil"
	"code.justin.tv/cb/dashy/view/multichannel"
	"golang.org/x/sync/errgroup"
)

// GET /v1/stats/multi_channel/sessions?channel_ids=
func (s *Server) v1MultiChannelSessions(w http.ResponseWriter, req *http.Request) {
	writer := httputil.NewJSONResponseWriter(w)
	channelIDs := req.Context().Value(contextKeyChannelIDs).([]int64)
	reqTimeRange := req.Context().Value(contextKeyTimeRange).(timeRange)
	startTime, endTime := reqTimeRange.startTime, reqTimeRange.endTime

	group, ctx := errgroup.WithContext(req.Context())
	mutex := sync.Mutex{}
	allChannelSessions := map[string][]multichannel.Session{}

	for _, channelID := range channelIDs {
		channelID := channelID

		group.Go(func() error {
			zephyrSessions, err := s.Zephyr.GetSessionsByTime(ctx, channelID, startTime, endTime)
			if err != nil {
				return err
			}

			mutex.Lock()
			allChannelSessions[strconv.FormatInt(channelID, 10)] = multiChannelSessions(zephyrSessions)
			mutex.Unlock()

			return nil
		})
	}

	if err := group.Wait(); err != nil {
		writer.InternalServerError("Failed to query sessions for one or more channels", err)
		return
	}

	response := multichannel.SessionsResponse{
		Status: http.StatusOK,
		Meta: multichannel.SessionsMeta{
			StartTime: &startTime,
			EndTime:   &endTime,
		},
		Data: allChannelSessions,
	}

	writer.OK(response)
}

func multiChannelSessions(zephyrSessions []zephyr.DynamoSession) []multichannel.Session {
	sessions := make([]multichannel.Session, len(zephyrSessions))

	for i := range zephyrSessions {
		sessions[i] = multichannel.Session{
			StartTime: &zephyrSessions[i].SegmentStartTime.Converted,
			EndTime:   &zephyrSessions[i].SegmentEndTime.Converted,
		}
	}

	return sessions
}
