package api

import (
	"net/http"

	"golang.org/x/sync/errgroup"

	"code.justin.tv/cb/dashy/internal/clients/zephyr"
	"code.justin.tv/cb/dashy/internal/httputil"
	"code.justin.tv/cb/dashy/internal/legal"
	"code.justin.tv/cb/dashy/view/multichannel"
)

// GET /v1/stats/multi_channel/video_play_demographics?channel_ids=&start_time=&end_time=
func (s *Server) v1MultiChannelVideoPlayDemographics(w http.ResponseWriter, req *http.Request) {
	writer := httputil.NewJSONResponseWriter(w)
	channelIDs := req.Context().Value(contextKeyChannelIDs).([]int64)
	reqTimeRange := req.Context().Value(contextKeyTimeRange).(timeRange)
	startTime, endTime := reqTimeRange.startTime, reqTimeRange.endTime

	geoFrequencyMaps := make([]map[string]int64, len(channelIDs))
	internalReferralFrequencyMaps := make([]map[string]int64, len(channelIDs))
	externalReferralFrequencyMaps := make([]map[string]int64, len(channelIDs))
	platformFrequencyMaps := make([]map[string]int64, len(channelIDs))
	totalsPerChannel := make([]int64, len(channelIDs))

	group, ctx := errgroup.WithContext(req.Context())

	for idx, channelID := range channelIDs {
		idx, channelID := idx, channelID

		group.Go(func() error {
			var sessions []zephyr.DynamoSession
			var err error

			subgroup, subCtx := errgroup.WithContext(ctx)

			sessions, err = s.Clients.Zephyr.GetSessionsByTime(subCtx, channelID, startTime, endTime)
			if err != nil {
				return err
			}

			subgroup.Go(func() error {
				subtotal, subtotalError := s.Clients.Zephyr.GetTotalViews(subCtx, sessions)
				if subtotalError != nil {
					return subtotalError
				}

				totalsPerChannel[idx] = subtotal
				return nil
			})

			subgroup.Go(func() error {
				timeSeries, err := s.Clients.Zephyr.GetVideoGeoMap(subCtx, sessions)
				if err != nil {
					return err
				}

				geoFrequencyMaps[idx] = reducedGeoFrequencyMap(timeSeries)
				return nil
			})

			subgroup.Go(func() error {
				timeSeries, err := s.Clients.Zephyr.GetVideoReferralMap(subCtx, sessions)
				if err != nil {
					return err
				}

				internal, external := reducedInternalAndExternalReferralFrequencyMaps(timeSeries)
				internalReferralFrequencyMaps[idx], externalReferralFrequencyMaps[idx] = internal, external
				return nil
			})

			subgroup.Go(func() error {
				timeSeries, err := s.Clients.Zephyr.GetVideoPlatformsMap(subCtx, sessions)
				if err != nil {
					return err
				}

				platformFrequencyMaps[idx] = reducedPlatformFrequencyMap(timeSeries)
				return nil
			})

			return subgroup.Wait()
		})
	}

	if err := group.Wait(); err != nil {
		writer.InternalServerError(err.Error(), err)
		return
	}

	response := multichannel.DemographicsResponse{
		Status: http.StatusOK,
		Meta: multichannel.DemographicsMeta{
			StartTime: &startTime,
			EndTime:   &endTime,
		},
		Data: legalizedMultiChannelDemographics(
			geoFrequencyMaps,
			internalReferralFrequencyMaps,
			externalReferralFrequencyMaps,
			platformFrequencyMaps,
			totalsPerChannel,
		),
	}

	writer.OK(response)
}

func reducedGeoFrequencyMap(geoMaps []zephyr.GeoMap) map[string]int64 {
	frequencies := map[string]int64{}

	for _, geoMap := range geoMaps {
		for country, count := range geoMap.Geo {
			frequencies[country] += count
		}
	}

	return frequencies
}

func reducedInternalAndExternalReferralFrequencyMaps(referralMaps []zephyr.ReferralMap) (map[string]int64, map[string]int64) {
	internalFrequencies := map[string]int64{}
	externalFrequencies := map[string]int64{}

	for _, referralMap := range referralMaps {
		for source, count := range referralMap.Internal {
			internalFrequencies[source] += count
		}

		for source, count := range referralMap.External {
			externalFrequencies[source] += count
		}
	}

	return internalFrequencies, externalFrequencies
}

func reducedPlatformFrequencyMap(platformMaps []zephyr.PlatformMap) map[string]int64 {
	frequencies := map[string]int64{}

	for _, platformMap := range platformMaps {
		for country, count := range platformMap.Platform {
			frequencies[country] += count
		}
	}

	return frequencies
}

func legalizedMultiChannelDemographics(geos, internals, externals, platforms []map[string]int64, totalViews []int64) multichannel.Demographics {
	geo := reducedStringToInt64Map(geos)
	var totalCount int64
	for _, subtotal := range totalViews {
		totalCount += subtotal
	}

	if !legal.DemographicsRevealed(totalCount) {
		return multichannel.Demographics{
			VideoPlayCount: totalCount,
			Geo:            map[string]int64{},
			Platform:       map[string]int64{},
			Referrer: multichannel.Referrer{
				Internal: map[string]int64{},
				External: map[string]int64{},
			},
		}
	}

	internal, external := legalizedReferrals(reducedStringToInt64Map(internals), reducedStringToInt64Map(externals), totalCount)

	uniqueGeoCount := 0
	for _, count := range geo {
		if count > 0 {
			uniqueGeoCount++
		}
	}

	return multichannel.Demographics{
		VideoPlayCount: totalCount,
		Geo:            legalizedGeo(geo, uniqueGeoCount, topGeoByFrequency(geo), totalCount),
		Platform:       legalizedPlatform(reducedStringToInt64Map(platforms), totalCount),
		Referrer: multichannel.Referrer{
			Internal:        internal,
			External:        external,
			InternalTwitch:  getInternalTwitchReferrers(internal),
			InternalChannel: getInternalChannelReferrers(internal),
		},
	}
}

func reducedStringToInt64Map(maps []map[string]int64) map[string]int64 {
	reduced := map[string]int64{}

	for _, singleMap := range maps {
		for key, val := range singleMap {
			reduced[key] += val
		}
	}

	return reduced
}

func topGeoByFrequency(geoFrequencies map[string]int64) string {
	var topGeo string
	var topFrequency int64

	for geo, frequency := range geoFrequencies {
		if frequency > topFrequency {
			topGeo = geo
			topFrequency = frequency
		}
	}

	return topGeo
}
