package api

import (
	"testing"

	"code.justin.tv/cb/dashy/internal/clients/zephyr"
	"code.justin.tv/cb/dashy/view/multichannel"

	"github.com/stretchr/testify/assert"
)

func TestReducedGeoFrequencyMap(t *testing.T) {
	geos := []zephyr.GeoMap{
		{
			Geo: map[string]int64{
				"A": 1,
				"B": 1,
			},
		},
		{
			Geo: map[string]int64{
				"B": 1,
				"C": 1,
			},
		},
		{
			Geo: map[string]int64{
				"C": 1,
				"D": 1,
			},
		},
	}

	expected := map[string]int64{
		"A": 1,
		"B": 2,
		"C": 2,
		"D": 1,
	}

	assert.Equal(t, expected, reducedGeoFrequencyMap(geos), "geo frequencies should be reduced")
}

func TestReducedInternalAndExternalReferralFrequencyMaps(t *testing.T) {
	referrals := []zephyr.ReferralMap{
		{
			Internal: map[string]int64{
				"internal A": 1,
				"internal B": 1,
			},
			External: map[string]int64{
				"external X": 1,
				"external Y": 1,
			},
		},
		{
			Internal: map[string]int64{
				"internal B": 1,
				"internal C": 1,
			},
		},
		{
			External: map[string]int64{
				"external Y": 1,
				"external Z": 1,
			},
		},
	}

	expectedInternals := map[string]int64{
		"internal A": 1,
		"internal B": 2,
		"internal C": 1,
	}

	expectedExternals := map[string]int64{
		"external X": 1,
		"external Y": 2,
		"external Z": 1,
	}

	internals, externals := reducedInternalAndExternalReferralFrequencyMaps(referrals)

	assert.Equal(t, expectedInternals, internals, "internal referral frequencies should be reduced")
	assert.Equal(t, expectedExternals, externals, "external referral frequencies should be reduced")
}

func TestReducedPlatformFrequencyMap(t *testing.T) {
	platforms := []zephyr.PlatformMap{
		{
			Platform: map[string]int64{
				"A": 1,
				"B": 1,
			},
		},
		{
			Platform: map[string]int64{
				"B": 1,
				"C": 1,
			},
		},
		{
			Platform: map[string]int64{
				"C": 1,
				"D": 1,
			},
		},
	}

	expected := map[string]int64{
		"A": 1,
		"B": 2,
		"C": 2,
		"D": 1,
	}

	assert.Equal(t, expected, reducedPlatformFrequencyMap(platforms), "platform frequencies should be reduced")
}

func TestLegalizedMultiChannelDemographics_DemographicsNotRevealed(t *testing.T) {
	geos := []map[string]int64{
		{"US": 1},
		{"AA": 3},
	}

	platforms := []map[string]int64{}
	internals := []map[string]int64{}
	externals := []map[string]int64{}
	views := []int64{4}

	expected := multichannel.Demographics{
		VideoPlayCount: 4,
		Geo:            map[string]int64{},
		Platform:       map[string]int64{},
		Referrer: multichannel.Referrer{
			Internal: map[string]int64{},
			External: map[string]int64{},
		},
	}

	demographics := legalizedMultiChannelDemographics(geos, internals, externals, platforms, views)

	assert.Equal(t, expected, demographics, "demographics should not be revealed")
}

func TestLegalizedMultiChannelDemographics_TopGeoOnly(t *testing.T) {
	geos := []map[string]int64{
		{"US": 2},
		{"AA": 3},
	}

	platforms := []map[string]int64{}
	internals := []map[string]int64{}
	externals := []map[string]int64{}
	views := []int64{5}

	expected := multichannel.Demographics{
		VideoPlayCount: 5,
		Geo: map[string]int64{
			"AA": 3,
		},
		Platform: map[string]int64{},
		Referrer: multichannel.Referrer{
			Internal:        map[string]int64{},
			External:        map[string]int64{},
			InternalTwitch:  map[string]int64{},
			InternalChannel: map[string]int64{},
		},
	}

	demographics := legalizedMultiChannelDemographics(geos, internals, externals, platforms, views)

	assert.Equal(t, expected, demographics, "geo demographics should contain top geo only")
}

func TestLegalizedMultiChannelDemographics_AggregatedPlatforms(t *testing.T) {
	geos := []map[string]int64{
		{"US": 2},
		{"AA": 3},
	}

	platforms := []map[string]int64{
		{platformAndroid: 2},
		{platformMobileWeb: 3},
	}

	internals := []map[string]int64{}
	externals := []map[string]int64{}
	views := []int64{5}

	expected := multichannel.Demographics{
		VideoPlayCount: 5,
		Geo: map[string]int64{
			"AA": 3,
		},
		Platform: map[string]int64{
			platformAggregate: 5,
		},
		Referrer: multichannel.Referrer{
			Internal:        map[string]int64{},
			External:        map[string]int64{},
			InternalTwitch:  map[string]int64{},
			InternalChannel: map[string]int64{},
		},
	}

	demographics := legalizedMultiChannelDemographics(geos, internals, externals, platforms, views)

	assert.Equal(t, expected, demographics, "platform demographics should be aggregated")
}

func TestLegalizedMultiChannelDemographics_AggregatedReferrers(t *testing.T) {
	geos := []map[string]int64{
		{"US": 2},
		{"AA": 3},
	}

	platforms := []map[string]int64{}

	internals := []map[string]int64{
		{"front_page_featured": 1},
		{"twitch": 1},
	}

	externals := []map[string]int64{
		{"google": 2},
		{"twitter": 1},
	}
	views := []int64{5}

	expected := multichannel.Demographics{
		VideoPlayCount: 5,
		Geo: map[string]int64{
			"AA": 3,
		},
		Platform: map[string]int64{},
		Referrer: multichannel.Referrer{
			Internal:        map[string]int64{},
			InternalTwitch:  map[string]int64{},
			InternalChannel: map[string]int64{},
			External: map[string]int64{
				referrerInternalAggregate: 2,
				referrerExternalAggregate: 3,
			},
		},
	}

	demographics := legalizedMultiChannelDemographics(geos, internals, externals, platforms, views)

	assert.Equal(t, expected, demographics, "referrer demographics should aggregate by internal and external")
}

func TestLegalizedMultiChannelDemographics_AggregatedChannelReferrers(t *testing.T) {
	geos := []map[string]int64{
		{"US": 6},
		{"AA": 4},
	}

	platforms := []map[string]int64{}

	internals := []map[string]int64{
		{"front_page_featured": 1},
		{"twitch": 9},
	}

	externals := []map[string]int64{}
	views := []int64{10}

	expected := multichannel.Demographics{
		VideoPlayCount: 10,
		Geo: map[string]int64{
			"US": 6,
		},
		Platform: map[string]int64{},
		Referrer: multichannel.Referrer{
			Internal: map[string]int64{
				"front_page_featured":            1,
				referrerInternalOtherChannelPage: 9,
			},
			InternalTwitch: map[string]int64{
				"front_page_featured": 1,
			},
			InternalChannel: map[string]int64{
				referrerInternalOtherChannelPage: 9,
			},
			External: map[string]int64{},
		},
	}

	demographics := legalizedMultiChannelDemographics(geos, internals, externals, platforms, views)

	assert.Equal(t, expected, demographics, "referrer demographics should aggregate channels")
}

func TestLegalizedMultiChannelDemographics_RevealAll(t *testing.T) {
	geos := []map[string]int64{
		{"US": 15},
		{"AA": 4},
		{"CA": 1},
	}

	platforms := []map[string]int64{
		{platformAndroid: 19},
		{platformMobileWeb: 1},
	}

	internals := []map[string]int64{
		{"front_page_featured": 1},
		{"twitch": 9},
	}

	externals := []map[string]int64{
		{"google": 10},
		{"twitter": 10},
	}
	views := []int64{20}

	expected := multichannel.Demographics{
		VideoPlayCount: 20,
		Geo: map[string]int64{
			"US": 15,
			"AA": 4,
			"CA": 1,
		},
		Platform: map[string]int64{
			platformAndroid:   19,
			platformMobileWeb: 1,
		},
		Referrer: multichannel.Referrer{
			Internal: map[string]int64{
				"front_page_featured": 1,
				"twitch":              9,
			},
			InternalTwitch: map[string]int64{
				"front_page_featured": 1,
			},
			InternalChannel: map[string]int64{
				"twitch": 9,
			},
			External: map[string]int64{
				"google":  10,
				"twitter": 10,
			},
		},
	}

	demographics := legalizedMultiChannelDemographics(geos, internals, externals, platforms, views)

	assert.Equal(t, expected, demographics, "all demographics should be revealed")
}
