package api

import (
	"context"
	"net/http"
	"time"

	"code.justin.tv/cb/dashy/internal/httputil"
	"code.justin.tv/cb/dashy/view/raid"
)

const (
	raidsStatsChunkSize = 5 // minutes
)

// v1RaidActivity returns the last broadcast raids's stats
// of a given channel ID within a specified time range.
func (s *Server) v1RaidActivity(w http.ResponseWriter, req *http.Request) {
	writer := httputil.NewJSONResponseWriter(w)
	channelID := req.Context().Value(contextKeyChannelID).(int64)

	reqTimeRange := req.Context().Value(contextKeyTimeRange).(timeRange)
	startTime, endTime := reqTimeRange.startTime, reqTimeRange.endTime

	raidTimeSeries, chunkSize, err := s.getRaidActivity(req.Context(), channelID, startTime, endTime)
	if err != nil {
		writer.InternalServerError("Failed to get raids counts", err)
		return
	}

	response := &raid.Response{
		Status: http.StatusOK,
		Meta: raid.Meta{
			StartTime:                 &startTime,
			EndTime:                   &endTime,
			SessionGapDurationMinutes: sessionMinuteGap,
			InterpolationChunkMinutes: chunkSize,
		},
		Data: raid.Data{
			Raid: raidTimeSeries,
		},
	}

	writer.OK(response)
}

// getRaidActivity returns the raids stats stats (using new tahoe processor)
// of a given channel ID within a specified time range.
func (s *Server) getRaidActivity(ctx context.Context, channelID int64, startTime time.Time, endTime time.Time) ([]*raid.RaidBreakdown, int, error) {
	// Query for all raids statistics within queried time range.
	raidsByTime, err := s.Clients.Zephyr.GetRaidsByTime(ctx, channelID, startTime, endTime)
	if err != nil {
		return nil, 0, err
	}

	// build the output raidTimeSeries by iterating raidsByTime
	// and converting the backend data into output data.
	raidTimeSeries := make([]*raid.RaidBreakdown, len(raidsByTime))
	for idx, raidObj := range raidsByTime {
		timestamp := raidObj.Time

		raidTimeSeries[idx] = &raid.RaidBreakdown{
			Timestamp:   &timestamp,
			ViewerCount: raidObj.ViewerCount,
			Count:       raidObj.Count,
			RaidID:      raidObj.RaidID,
		}

	}

	return raidTimeSeries, raidsStatsChunkSize, err
}
