package api

import (
	"net/http"
	"time"

	"code.justin.tv/cb/dashy/internal/auth"
	"code.justin.tv/cb/dashy/internal/clients"
	"code.justin.tv/cb/dashy/internal/httputil"
	"code.justin.tv/cb/dashy/internal/statsd"
	"goji.io"
	"goji.io/pat"
)

// Dashy's SLA - max requests duration
const handlerTimeout = 5 * time.Second

// Server contains a router, clients, and environment-dependent utilities.
type Server struct {
	mux         *goji.Mux
	authDecoder *auth.Decoder
	*clients.Clients
	StatsD *statsd.Client
}

// ServerParams is passed to NewServer to instantiate a Server.
type ServerParams struct {
	AuthDecoder *auth.Decoder
	Clients     *clients.Clients
	StatsD      *statsd.Client
}

// NewServer instantiates a Server with the defined routes and corresponding handlers,
// and returns the Server.
func NewServer(params *ServerParams) *Server {
	server := &Server{
		mux:         goji.NewMux(),
		authDecoder: params.AuthDecoder,
		Clients:     params.Clients,
		StatsD:      params.StatsD,
	}

	server.mux.Use(httputil.PanicRecoverer)
	server.mux.HandleFunc(pat.Get("/health"), httputil.HealthCheck)

	root := goji.SubMux()
	server.mux.Handle(pat.New("/*"), root)
	root.Use(httputil.AccessLogger)
	root.Use(server.StatsD.MetricCollector)
	root.Use(httputil.Timeout(handlerTimeout))

	v1 := goji.SubMux()
	root.Handle(pat.New("/v1/*"), v1)

	dashboard := goji.SubMux()
	v1.Handle(pat.New("/dashboard/*"), dashboard)
	dashboardChannels := goji.SubMux()
	dashboard.Handle(pat.New("/channels/:channel_id/*"), dashboardChannels)
	dashboardChannels.Use(validateChannelID)
	dashboardChannels.Use(server.authorizeChannelViewStats)
	dashboardChannels.HandleFunc(pat.Get("/layout"), server.v1Layout)
	dashboardChannels.HandleFunc(pat.Get("/tag_filters"), server.v1TagFilters)

	stats := goji.SubMux()
	v1.Handle(pat.New("/stats/*"), stats)

	statsMultiChannel := goji.SubMux()
	stats.Handle(pat.New("/multi_channel/*"), statsMultiChannel)
	statsMultiChannel.Use(server.authorizeMultiChannelViewStats)
	statsMultiChannel.Use(validateQueryParamChannelIDs)
	statsMultiChannel.Handle(pat.Get("/chat_activities"), validateTimeRange(server.v1MultiChannelChatActivities))
	statsMultiChannel.Handle(pat.Get("/commercials"), validateTimeRange(server.v1MultiChannelCommercials))
	statsMultiChannel.Handle(pat.Get("/concurrent_viewers"), validateTimeRange(server.v1MultiChannelConcurrentViewers))
	statsMultiChannel.Handle(pat.Get("/follows"), validateTimeRange(server.v1MultiChannelFollows))
	statsMultiChannel.Handle(pat.Get("/sessions"), validateTimeRange(server.v1MultiChannelSessions))
	statsMultiChannel.Handle(pat.Get("/video_play_demographics"), validateTimeRange(server.v1MultiChannelVideoPlayDemographics))

	statsChannels := goji.SubMux()
	stats.Handle(pat.New("/channels/:channel_id/*"), statsChannels)
	statsChannels.Use(validateChannelID)
	statsChannels.Use(server.authorizeChannelViewStats)
	statsChannels.Handle(pat.Get("/chat_activity"), validateTimeRange(server.v1ChatActivity))
	statsChannels.Handle(pat.Get("/clips"), validateTimeRange(server.v1Clips))
	statsChannels.Handle(pat.Get("/commercials"), validateTimeRange(server.v1Commercials))
	statsChannels.Handle(pat.Get("/followers"), validateTimeRange(server.v1Followers))
	statsChannels.Handle(pat.Get("/sessions_summary"), validateTimeRange(server.v1SessionsSummary))
	statsChannels.Handle(pat.Get("/stats"), validateTimeRange(server.v1Stats))
	statsChannels.Handle(pat.Get("/video_play_demographics"), validateTimeRange(server.v1VideoPlayDemographics))
	statsChannels.Handle(pat.Get("/views_summary"), validateTimeRange(server.v1UniqueViews))
	statsChannels.Handle(pat.Get("/raids_activity"), validateTimeRange(server.v1RaidActivity))
	statsChannels.Handle(pat.Get("/subscription_purchase"), validateTimeRange(server.v1SubscriptionPurchaseActivity))

	v1.HandleFunc(pat.Get("/stream_summary/tip"), server.v1Tip)

	return server
}

// ServeHTTP allows Server to implement http.Handler.
func (s *Server) ServeHTTP(w http.ResponseWriter, req *http.Request) {
	s.mux.ServeHTTP(w, req)
}
