package api

import (
	"net/http"
	"time"

	"fmt"

	"code.justin.tv/cb/dashy/internal/clients/dynamodb"
	"code.justin.tv/cb/dashy/internal/httputil"
	"code.justin.tv/cb/dashy/view/summary"
	log "github.com/sirupsen/logrus"
	"golang.org/x/sync/errgroup"
)

type SummaryResponses struct {
	Sessions  []*summary.Session
	Analytics []dynamodb.NotificationsAnalytics
}

// v1SessionsSummary returns a collection of broadcast sessions and summary data
// given a channelID and a time range
func (s *Server) v1SessionsSummary(w http.ResponseWriter, req *http.Request) {
	writer := httputil.NewJSONResponseWriter(w)
	channelID := req.Context().Value(contextKeyChannelID).(int64)

	reqTimeRange := req.Context().Value(contextKeyTimeRange).(timeRange)
	startTime, endTime := reqTimeRange.startTime, reqTimeRange.endTime

	responses := SummaryResponses{}
	g := errgroup.Group{}
	g.Go(func() error {
		sessions, err := s.Zephyr.GetSessionsSummaryByTime(req.Context(), channelID, startTime, endTime)
		if err != nil {
			log.WithFields(log.Fields{
				"channelID": channelID,
				"startTime": startTime.Format(time.RFC3339),
				"endTime":   endTime.Format(time.RFC3339),
			}).WithError(err).Error("failed to GetSessionsSummaryByTime")
			return err
		}

		responses.Sessions = sessions
		return nil
	})

	g.Go(func() error {
		notificationsAnalytics, err := s.Clients.DynamoDB.NotificationsAnalytics.GetAllByChannelIDAndTime(req.Context(), fmt.Sprintf("%d", channelID), startTime, endTime)
		if err != nil {
			log.WithFields(log.Fields{
				"channelID": channelID,
				"startTime": startTime.Format(time.RFC3339),
				"endTime":   endTime.Format(time.RFC3339),
			}).WithError(err).Error("failed to NotificationsAnalytics.GetAllByChannelIDAndTime")
			return err
		}

		responses.Analytics = notificationsAnalytics
		return nil
	})

	if err := g.Wait(); err != nil {
		writer.InternalServerError("Failed to get session summary or notifications analytics: ", err)
	}

	mergeSummaries(responses.Sessions, responses.Analytics)

	response := &summary.Response{
		Status: http.StatusOK,
		Meta: summary.Meta{
			StartTime: &startTime,
			EndTime:   &endTime,
		},
		Sessions: responses.Sessions,
	}

	writer.OK(response)
}

func mergeSummaries(sessions []*summary.Session, notificationsAnalytics []dynamodb.NotificationsAnalytics) {
	if sessions == nil {
		return
	}

	broadcastIDToAnalytics := map[string]dynamodb.NotificationsAnalytics{}
	for _, analytic := range notificationsAnalytics {
		broadcastIDToAnalytics[analytic.BroadcastID] = analytic
	}

	for _, session := range sessions {
		sessionAnalytics := map[string]summary.NotificationsAnalytics{}
		for _, broadcastID := range session.BroadcastIDs {
			if sessionAnalytic, ok := broadcastIDToAnalytics[broadcastID]; ok {
				sessionAnalytics[broadcastID] = dynamodbToSummaryNotificationsAnalytics(sessionAnalytic)
			}
		}

		session.NotificationAnalytics = sessionAnalytics
	}
}

func dynamodbToSummaryNotificationsAnalytics(notificationsAnalytics dynamodb.NotificationsAnalytics) summary.NotificationsAnalytics {
	resp := summary.NotificationsAnalytics{
		ChannelID:          notificationsAnalytics.ChannelID,
		BroadcastID:        notificationsAnalytics.BroadcastID,
		StartTime:          notificationsAnalytics.StartTime,
		FollowerCount:      notificationsAnalytics.FollowerCount,
		EngagementCount:    notificationsAnalytics.EngagementCount,
		NotificationCount:  notificationsAnalytics.NotificationCount,
		CustomNotification: notificationsAnalytics.CustomNotification,
	}

	return resp
}
