package api

import (
	"math"
	"net/http"
	"time"

	"code.justin.tv/cb/dashy/internal/httputil"
	"code.justin.tv/cb/dashy/view/stats"
)

const (
	concurrentsStatsChunkSize             = 5 // minutes
	concurrentsInterpolationPointEstimate = 85
)

// v1Stats returns the last broadcast session's stats
// of a given channel ID within a specified time range.
func (s *Server) v1Stats(w http.ResponseWriter, req *http.Request) {
	writer := httputil.NewJSONResponseWriter(w)
	ctx := req.Context()
	channelID := ctx.Value(contextKeyChannelID).(int64)

	reqTimeRange := req.Context().Value(contextKeyTimeRange).(timeRange)
	startTime, endTime := reqTimeRange.startTime, reqTimeRange.endTime

	concurrents, err := s.Clients.Zephyr.GetConcurrentViewersByTime(ctx, channelID, startTime, endTime)
	if err != nil {
		writer.InternalServerError("Failed to get concurrent viewer counts", err)
		return
	}

	raw := make(rawConcurrentViewers, len(concurrents))
	for idx := 0; idx < len(concurrents); idx++ {
		raw[idx] = &stats.ConcurrentViewerCount{
			Timestamp: &concurrents[idx].Timestamp,
			Count:     concurrents[idx].AverageCount,
		}
	}

	interpolatedConcurrents, chunkSize := raw.interpolate(concurrentsInterpolationPointEstimate)

	response := &stats.Response{
		Status: http.StatusOK,
		Meta: stats.Meta{
			StartTime:                 &startTime,
			EndTime:                   &endTime,
			SessionGapDurationMinutes: sessionMinuteGap,
			InterpolationChunkMinutes: chunkSize,
		},
		Data: stats.Data{
			ConcurrentViewers: interpolatedConcurrents,
		},
	}

	writer.OK(response)
}

type rawConcurrentViewers []*stats.ConcurrentViewerCount

// take a data response object and reduce the number of datapoints such
//	that it is less than {points}
func (data rawConcurrentViewers) interpolate(points int) ([]*stats.ConcurrentViewerCount, int) {
	// if there arent many points to begin with we dont need to
	//	interpolate
	pointCount := len(data)
	if pointCount < points {
		return data, concurrentsStatsChunkSize
	}

	interpolated := []*stats.ConcurrentViewerCount{}
	dataRange := data[pointCount-1].Timestamp.Sub(*data[0].Timestamp)
	segmentDuration := time.Duration(concurrentsStatsChunkSize) * time.Minute

	// the number of chunks we should combine to reduce the total number of points
	//	around the number the user requested
	chunkCount := math.Ceil(dataRange.Seconds() / segmentDuration.Seconds() / float64(points))
	// the size of each chunk in minutes
	chunkDurationInt := int(chunkCount * concurrentsStatsChunkSize)
	chunkDuration := time.Duration(chunkDurationInt) * time.Minute

	chunkCutoff := data[0].Timestamp.Add(chunkDuration)
	chunkCurrent := &stats.ConcurrentViewerCount{
		Count:     data[0].Count,
		Timestamp: data[0].Timestamp,
	}
	// number of points accumulated in each chunk
	numChunkPoints := float64(1)
	idx := 1

	for idx < pointCount {
		if !data[idx].Timestamp.Before(chunkCutoff) {
			// for concurrent viewers we want the average vewers in the new sample size
			if numChunkPoints > 0 {
				chunkCurrent.Count = chunkCurrent.Count / numChunkPoints
				interpolated = append(interpolated, chunkCurrent)
			}
			chunkTimestamp := chunkCutoff

			numChunkPoints = 0
			chunkCurrent = &stats.ConcurrentViewerCount{
				Count:     0,
				Timestamp: &chunkTimestamp,
			}

			chunkCutoff = chunkCutoff.Add(chunkDuration)

			continue
		}

		numChunkPoints++
		chunkCurrent.Count += data[idx].Count
		idx++
	}

	if numChunkPoints > 0 {
		chunkCurrent.Count = chunkCurrent.Count / numChunkPoints
		interpolated = append(interpolated, chunkCurrent)
	}

	return interpolated, chunkDurationInt
}
