package api

import (
	"context"
	"net/http"
	"time"

	"code.justin.tv/cb/dashy/internal/httputil"
	"code.justin.tv/cb/dashy/view/subscriptionpurchase"
)

const (
	subscriptionPurchaseSuccessStatsChunkSize = 5 // minutes
)

// v1SubscriptionPurchaseActivity returns the last broadcast subscription purchase success's stats
// of a given channel ID within a specified time range.
func (s *Server) v1SubscriptionPurchaseActivity(w http.ResponseWriter, req *http.Request) {
	writer := httputil.NewJSONResponseWriter(w)
	channelID := req.Context().Value(contextKeyChannelID).(int64)

	reqTimeRange := req.Context().Value(contextKeyTimeRange).(timeRange)
	startTime, endTime := reqTimeRange.startTime, reqTimeRange.endTime

	subscriptionPurchaseTimeSeries, chunkSize, err := s.getSubscriptionPurchaseActivity(req.Context(), channelID, startTime, endTime)

	if err != nil {
		writer.InternalServerError("Failed to get subscription purchase success counts", err)
		return
	}

	response := &subscriptionpurchase.Response{
		Status: http.StatusOK,
		Meta: subscriptionpurchase.Meta{
			StartTime:                 &startTime,
			EndTime:                   &endTime,
			SessionGapDurationMinutes: sessionMinuteGap,
			InterpolationChunkMinutes: chunkSize,
		},
		Data: subscriptionpurchase.Data{
			SubscriptionPurchase: subscriptionPurchaseTimeSeries,
		},
	}

	writer.OK(response)
}

// getSubscriptionPurchaseActivity returns the subscription purchase stats from Zephyr
// of a given channel ID within a specified time range.
func (s *Server) getSubscriptionPurchaseActivity(ctx context.Context, channelID int64, startTime time.Time, endTime time.Time) ([]*subscriptionpurchase.SubscriptionPurchaseBreakdown, int, error) {
	// Query for all subscription purchase statistics within queried time range.
	subscriptionPurchaseByTime, err := s.Clients.Zephyr.GetSubscriptionPurchaseSuccessByTime(ctx, channelID, startTime, endTime)
	if err != nil {
		return nil, 0, err
	}

	// build the output subscriptionPurchaseTimeSeries by iterating subscriptionPurchaseByTime
	// and converting the backend data into output data.
	subscriptionPurchaseTimeSeries := make([]*subscriptionpurchase.SubscriptionPurchaseBreakdown, len(subscriptionPurchaseByTime))
	for idx, subscriptionPurchaseSuccessObj := range subscriptionPurchaseByTime {
		timestamp := subscriptionPurchaseSuccessObj.Time

		subscriptionPurchaseTimeSeries[idx] = &subscriptionpurchase.SubscriptionPurchaseBreakdown{
			Timestamp: &timestamp,
			Count:     subscriptionPurchaseSuccessObj.Count,
		}

	}

	return subscriptionPurchaseTimeSeries, subscriptionPurchaseSuccessStatsChunkSize, err
}
