package api

import (
	"math/rand"
	"net/http"
	"strings"
	"time"

	log "github.com/sirupsen/logrus"

	"code.justin.tv/cb/dashy/internal/clients/dynamodb"
	"code.justin.tv/cb/dashy/internal/httputil"
	"code.justin.tv/cb/dashy/view/tips"
)

var validLanguages = map[string]struct{}{
	"ar":    {},
	"bg":    {},
	"cs":    {},
	"da":    {},
	"de":    {},
	"el":    {},
	"en":    {},
	"es":    {},
	"es-mx": {},
	"fi":    {},
	"fr":    {},
	"hu":    {},
	"it":    {},
	"ja":    {},
	"ko":    {},
	"nl":    {},
	"no":    {},
	"pl":    {},
	"pt":    {},
	"pt-br": {},
	"ru":    {},
	"sk":    {},
	"sv":    {},
	"th":    {},
	"tr":    {},
	"vi":    {},
	"zh-cn": {},
	"zh-tw": {},
	"zh":    {},
}

const defaultLanguage = "en"

var fallbackTip = dynamodb.Tip{
	Body: "Remember to always have fun - your personality and energy will draw viewers to your stream.",
}
var seededRand *rand.Rand = rand.New(rand.NewSource(time.Now().UTC().UnixNano()))

//
// V1
//

// v1Tip returns a tip.
func (s *Server) v1Tip(w http.ResponseWriter, req *http.Request) {
	writer := httputil.NewJSONResponseWriter(w)

	language := req.Header.Get("Accept-Language")

	// Visage locale parsing uses https://godoc.org/golang.org/x/text/language#Tag which
	// auto-capitalizes regions, e.g. pt-br actually appears as pt-BR.
	// Convert these back to lowercase
	language = strings.ToLower(language)

	if _, ok := validLanguages[language]; !ok {
		language = defaultLanguage
	}

	fetchedTips, err := s.DynamoDB.Tips.GetAllByKey(req.Context(), language)
	if err != nil {
		log.WithError(err).Error("Dynamo failed to respond when fetching tip")
		fetchedTips = []dynamodb.Tip{fallbackTip}
	} else if len(fetchedTips) == 0 {
		fetchedTips = []dynamodb.Tip{fallbackTip}
	}

	tip := fetchedTips[seededRand.Intn(len(fetchedTips))]

	response := &tips.Response{
		Status: http.StatusOK,
		Data: tips.Data{
			Tip: tips.Tip{
				Body: tip.Body,
				URL:  tip.URL,
			},
		},
	}

	w.Header().Set("Content-Language", language)
	writer.OK(response)
}
