package api

import (
	"context"
	"net/http"
	"time"

	"code.justin.tv/cb/dashy/internal/httputil"
	"code.justin.tv/cb/dashy/view/uniqueviews"
)

const (
	uniqueViewsStatsChunkSize = 5 // minutes
)

// v1UniqueViews returns the unique views stats
// of a given channel ID within a specified time range.
func (s *Server) v1UniqueViews(w http.ResponseWriter, req *http.Request) {
	writer := httputil.NewJSONResponseWriter(w)
	channelID := req.Context().Value(contextKeyChannelID).(int64)

	reqTimeRange := req.Context().Value(contextKeyTimeRange).(timeRange)
	startTime, endTime := reqTimeRange.startTime, reqTimeRange.endTime

	uniqueViewTimeSeries, chunkSize, err := s.getUniqueViews(req.Context(), channelID, startTime, endTime)
	if err != nil {
		writer.InternalServerError("Failed to get unique views activity counts", err)
		return
	}

	response := &uniqueviews.Response{
		Status: http.StatusOK,
		Meta: uniqueviews.Meta{
			StartTime:                 &startTime,
			EndTime:                   &endTime,
			SessionGapDurationMinutes: sessionMinuteGap,
			InterpolationChunkMinutes: chunkSize,
		},
		Data: uniqueviews.Data{
			UniqueView: uniqueViewTimeSeries,
		},
	}

	writer.OK(response)
}

// getUniqueViews returns the last broadcast session's stats (using zephyr)
// of a given channel ID within a specified time range.
func (s *Server) getUniqueViews(ctx context.Context, channelID int64, startTime time.Time, endTime time.Time) ([]*uniqueviews.UniqueViewBreakdown, int, error) {
	// Query for all chat statistics within queried time range.
	uniqueViewsByTime, err := s.Clients.Zephyr.GetViewsByTime(ctx, channelID, startTime, endTime)
	if err != nil {
		return nil, 0, err
	}

	// build the output uniqueViewTimeSeries by iterating uniqueViewsByTime
	// and converting the backend data into output data.
	uniqueViewTimeSeries := make([]*uniqueviews.UniqueViewBreakdown, len(uniqueViewsByTime))
	for idx, uniqueViews := range uniqueViewsByTime {
		timestamp := uniqueViews.Time

		uniqueViewTimeSeries[idx] = &uniqueviews.UniqueViewBreakdown{
			Timestamp:   &timestamp,
			TotalViews:  uniqueViews.TotalViews,
			UniqueViews: uniqueViews.UniqueViews,
		}
	}

	return uniqueViewTimeSeries, uniqueViewsStatsChunkSize, err
}
