package api

import (
	"fmt"
	"net/http"
	"time"

	"golang.org/x/sync/errgroup"

	"code.justin.tv/cb/dashy/internal/clients/zephyr"
	"code.justin.tv/cb/dashy/internal/httputil"
	"code.justin.tv/cb/dashy/internal/legal"
	"code.justin.tv/cb/dashy/internal/set"
	"code.justin.tv/cb/dashy/view/vpdemographics"
)

const (
	platformAndroid   = "android"
	platformIOS       = "ios"
	platformMobileWeb = "mobile_web"
	platformAggregate = "mobile_aggregated"

	referrerInternalOtherChannelPage = "other_channel_page"
	referrerInternalAggregate        = "twitch_aggregate"
	referrerExternalAggregate        = "external_aggregate"
)

var internalReferrerWhitelist = map[string]bool{
	"front_page_featured":      true,
	"creative_page_featured":   true,
	"hosted":                   true,
	"email_live_notification":  true,
	"onsite_notification":      true,
	"followed_channel":         true,
	"directory_browse":         true,
	"top_nav_bar":              true,
	"recommended_channel":      true,
	"search":                   true,
	"clips_live":               true,
	"friend_presence":          true,
	"homepage_carousel":        true,
	"homepage_recommendations": true,
	"other":                    true,
}

// v1VideoPlayDemographics returns the last broadcast session's stats
// of a given channel ID within a specified time range.
func (s *Server) v1VideoPlayDemographics(w http.ResponseWriter, req *http.Request) {
	writer := httputil.NewJSONResponseWriter(w)
	channelID := req.Context().Value(contextKeyChannelID).(int64)

	reqTimeRange := req.Context().Value(contextKeyTimeRange).(timeRange)
	startTime, endTime := reqTimeRange.startTime, reqTimeRange.endTime

	sessions, sessionsErr := s.Zephyr.GetSessionsByTime(req.Context(), channelID, startTime, endTime)
	if sessionsErr != nil {
		writer.InternalServerError(sessionsErr.Error(), sessionsErr)
		return
	}

	var geographical []zephyr.GeoMap
	var referral []zephyr.ReferralMap
	var platform []zephyr.PlatformMap
	var err error
	group, ctx := errgroup.WithContext(req.Context())

	// Query for geographical demographics
	group.Go(func() error {
		geographical, err = s.Zephyr.GetVideoGeoMap(ctx, sessions)
		if err != nil {
			return fmt.Errorf("Failed to get geographical demographics: %s", err)
		}

		return nil
	})

	// Query for referral demographics
	group.Go(func() error {
		referral, err = s.Zephyr.GetVideoReferralMap(ctx, sessions)
		if err != nil {
			return fmt.Errorf("Failed to get referral demographics: %s", err)
		}

		return nil
	})

	// Query for platform demographics
	group.Go(func() error {
		platform, err = s.Zephyr.GetVideoPlatformsMap(ctx, sessions)
		if err != nil {
			return fmt.Errorf("Failed to get platform demographics: %s", err)
		}

		return nil
	})

	if err := group.Wait(); err != nil {
		writer.InternalServerError(err.Error(), err)
		return
	}

	// total number of video-play events, we choose geo because it has the fewest amount of restrictions
	totalPlays := int64(0)
	// unique countries video-play events are from
	countries := set.NewStringSet()
	countryFrequency := map[string]int64{}
	countryCountMax := int64(0)
	var topCountry string
	aggregated := []*vpdemographics.VideoPlayDemographics{}

	for _, geoMap := range geographical {
		for country, count := range geoMap.Geo {
			totalPlays += count
			countries.Add(country)

			if _, ok := countryFrequency[country]; !ok {
				countryFrequency[country] = count
			} else {
				countryFrequency[country] += count
			}

			if countryFrequency[country] > countryCountMax {
				topCountry = country
				countryCountMax = countryFrequency[country]
			}
		}
	}

	// due to legal reasons if there are less than 'minimumVideoPlayCountForRevealingAnyData' video play events we return nothing
	if legal.DemographicsRevealed(totalPlays) {
		// merge all three timeseries into one timeseries
		geographicalIdx := 0
		platformIdx := 0
		referralIdx := 0
		isGeographicalDataNotEmpty := len(geographical) > 0
		isReferralDataNotEmpty := len(referral) > 0
		isPlatformDataNotEmpty := len(platform) > 0

		//while all three arrays arent empty
		for isGeographicalDataNotEmpty ||
			isReferralDataNotEmpty ||
			isPlatformDataNotEmpty {

			//create an empty datapoint
			current := vpdemographics.VideoPlayDemographics{}

			//get the earliest value for demo
			timestamps := []time.Time{}
			if isGeographicalDataNotEmpty {
				timestamps = append(timestamps, geographical[geographicalIdx].Time.Converted)
			}
			if isPlatformDataNotEmpty {
				timestamps = append(timestamps, platform[platformIdx].Time.Converted)
			}
			if isReferralDataNotEmpty {
				timestamps = append(timestamps, referral[referralIdx].Time.Converted)
			}

			//calculate the earliest timestamp from the three
			min := getMinTime(timestamps)
			current.Timestamp = &min

			//if the timestamp from the earliest value from ea demo matches the minimum timestamp
			//	then add the value to the appropriate field
			if isGeographicalDataNotEmpty && geographical[geographicalIdx].Time.Converted.Equal(min) {
				current.Geographical = legalizedGeo(geographical[geographicalIdx].Geo, countries.Length(), topCountry, totalPlays)

				geographicalIdx++
				isGeographicalDataNotEmpty = geographicalIdx < len(geographical)
			}

			if isPlatformDataNotEmpty && platform[platformIdx].Time.Converted.Equal(min) {
				current.Platform = legalizedPlatform(platform[platformIdx].Platform, totalPlays)

				platformIdx++
				isPlatformDataNotEmpty = platformIdx < len(platform)
			}

			if isReferralDataNotEmpty && referral[referralIdx].Time.Converted.Equal(min) {
				internal, external := legalizedReferrals(referral[referralIdx].Internal, referral[referralIdx].External, totalPlays)

				current.Referral = vpdemographics.ReferralBreakdown{
					Internal:        internal,
					External:        external,
					InternalTwitch:  getInternalTwitchReferrers(internal),
					InternalChannel: getInternalChannelReferrers(internal),
				}

				referralIdx++
				isReferralDataNotEmpty = referralIdx < len(referral)
			}

			aggregated = append(aggregated, &current)
		}
	}

	response := &vpdemographics.Response{
		Status: http.StatusOK,
		Meta: vpdemographics.Meta{
			StartTime:                 &startTime,
			EndTime:                   &endTime,
			SessionGapDurationMinutes: sessionMinuteGap,
			VideoPlayCount:            totalPlays,
		},
		Data: vpdemographics.Data{
			Demographics: aggregated,
		},
	}

	writer.OK(response)
}

func getMinTime(arr []time.Time) time.Time {
	min := arr[0]
	for _, t := range arr {
		if t.Before(min) {
			min = t
		}
	}

	return min
}

func legalizedGeo(geos map[string]int64, uniqueGeoCount int, topGeo string, totalPlays int64) map[string]int64 {
	legalCountrySet := map[string]int64{}
	if !(legal.AnyCountryRevealed(totalPlays)) {
		return legalCountrySet
	}
	if !(legal.InvididualCountriesRevealed(uniqueGeoCount)) {
		if count, ok := geos[topGeo]; ok {
			legalCountrySet[topGeo] = count
		}
		return legalCountrySet
	}
	return geos
}

func legalizedPlatform(platforms map[string]int64, playCount int64) map[string]int64 {
	if legal.IndividualPlatformRevealed(playCount) {
		return platforms
	}

	// Aggregate Android, iOS, and mobile web:
	var aggregate int64

	aggregate += platforms[platformAndroid]
	aggregate += platforms[platformIOS]
	aggregate += platforms[platformMobileWeb]

	if aggregate > 0 {
		platforms[platformAggregate] += aggregate
	}

	delete(platforms, platformAndroid)
	delete(platforms, platformIOS)
	delete(platforms, platformMobileWeb)

	return platforms
}

func legalizedReferrals(internals, externals map[string]int64, playCount int64) (map[string]int64, map[string]int64) {
	if !legal.IndividualReferralRevealed(playCount) {
		var internalAggregateCount int64
		var externalAggregateCount int64
		aggregatedExternals := map[string]int64{}

		for _, count := range internals {
			internalAggregateCount += count
		}

		for _, count := range externals {
			externalAggregateCount += count
		}

		// Both referrerInternalAggregate and referrerExternalAggregate are categorized as external
		// because we legally treat Twitch as an external domain for obfuscation.
		if internalAggregateCount > 0 {
			aggregatedExternals[referrerInternalAggregate] = internalAggregateCount
		}

		if externalAggregateCount > 0 {
			aggregatedExternals[referrerExternalAggregate] = externalAggregateCount
		}

		return map[string]int64{}, aggregatedExternals
	}

	if !legal.ChannelNameRevealed(playCount) {
		var channelAggregateCount int64
		internalsWithoutChannelNames := map[string]int64{}

		for name, count := range internals {
			if _, ok := internalReferrerWhitelist[name]; ok {
				internalsWithoutChannelNames[name] = count
			} else {
				channelAggregateCount += count
			}
		}

		if channelAggregateCount > 0 {
			internalsWithoutChannelNames[referrerInternalOtherChannelPage] = channelAggregateCount
		}

		return internalsWithoutChannelNames, externals
	}

	return internals, externals
}

func getInternalTwitchReferrers(internals map[string]int64) map[string]int64 {
	filteredReferrers := map[string]int64{}
	for referrer := range internals {
		if _, ok := internalReferrerWhitelist[referrer]; ok {
			filteredReferrers[referrer] = internals[referrer]
		}
	}
	return filteredReferrers
}

func getInternalChannelReferrers(internals map[string]int64) map[string]int64 {
	filteredReferrers := map[string]int64{}
	for referrer := range internals {
		if _, ok := internalReferrerWhitelist[referrer]; !ok {
			filteredReferrers[referrer] = internals[referrer]
		}
	}
	return filteredReferrers
}
