package api

import (
	"testing"

	"github.com/stretchr/testify/assert"
)

func TestLegalizedGeo_AllGeosRevealed(t *testing.T) {
	uniqueGeoCount := 3
	topGeo := "US"

	geos := map[string]int64{
		"AA": 4,
		"ZZ": 10,
	}

	var totalPlays int64 = 20

	assert.Equal(t, geos, legalizedGeo(geos, uniqueGeoCount, topGeo, totalPlays), "geos should be individualized")
}

func TestLegalizedGeo_TopGeoOnly(t *testing.T) {
	uniqueGeoCount := 2
	topGeo := "US"

	geos := map[string]int64{
		"US": 4,
		"ZZ": 10,
	}

	expected := map[string]int64{
		"US": 4,
	}

	var totalPlays int64 = 4

	assert.Equal(t, expected, legalizedGeo(geos, uniqueGeoCount, topGeo, totalPlays), "geos should only contain top country")
}

func TestLegalizedGeo_EmptyLowPlays(t *testing.T) {
	uniqueGeoCount := 5
	topGeo := "US"

	geos := map[string]int64{
		"AA": 4,
		"ZZ": 10,
	}

	expected := map[string]int64{}

	var totalPlays int64

	assert.Equal(t, expected, legalizedGeo(geos, uniqueGeoCount, topGeo, totalPlays), "geos should all be removed")
}

func TestLegalizedGeo_Empty(t *testing.T) {
	uniqueGeoCount := 2
	topGeo := "US"

	geos := map[string]int64{
		"AA": 4,
		"ZZ": 10,
	}

	expected := map[string]int64{}

	var totalPlays int64 = 10

	assert.Equal(t, expected, legalizedGeo(geos, uniqueGeoCount, topGeo, totalPlays), "geos should all be removed")
}

func TestLegalizedPlatform_AllPlatformsRevealed(t *testing.T) {
	var playCount int64 = 10

	platforms := map[string]int64{
		platformAndroid: 5,
		platformIOS:     5,
	}

	assert.Equal(t, platforms, legalizedPlatform(platforms, playCount), "platforms should be individualized")
}

func TestLegalizedPlatform_MobileAggregated(t *testing.T) {
	var playCount int64 = 9

	platforms := map[string]int64{
		platformAndroid:   1,
		platformMobileWeb: 1,
	}

	expected := map[string]int64{
		platformAggregate: 2,
	}

	assert.Equal(t, expected, legalizedPlatform(platforms, playCount), "platforms should be aggregated")
}

func TestLegalizedPlatform_NoMobileAggregated(t *testing.T) {
	var playCount int64 = 9

	platforms := map[string]int64{
		"web":           8,
		platformAndroid: 0,
	}

	expected := map[string]int64{
		"web": 8,
	}

	assert.Equal(t, expected, legalizedPlatform(platforms, playCount), "platforms should exclude mobile platforms if zero")
}

func TestLegalizedReferrals_ExternalAggregatesOnly(t *testing.T) {
	var playCount int64 = 9

	internalReferrals := map[string]int64{
		"my channel":      1,
		"another channel": 2,
	}

	externalReferrals := map[string]int64{
		"google":  3,
		"twitter": 3,
	}

	expectdInternals := map[string]int64{}

	expectedExternals := map[string]int64{
		referrerInternalAggregate: 3,
		referrerExternalAggregate: 6,
	}

	legalizedInternals, legalizedExternals := legalizedReferrals(internalReferrals, externalReferrals, playCount)

	assert.Equal(t, expectdInternals, legalizedInternals, "internals should be empty map")
	assert.Equal(t, expectedExternals, legalizedExternals, "externals should contain internal and external aggregates")
}

func TestLegalizedReferrals_ChannelNamesAggregated(t *testing.T) {
	var playCount int64 = 10

	internalReferrals := map[string]int64{
		"front_page_featured": 1,
		"my channel":          1,
		"another channel":     2,
	}

	externalReferrals := map[string]int64{
		"google":  3,
		"twitter": 3,
	}

	expectdInternals := map[string]int64{
		"front_page_featured":            1,
		referrerInternalOtherChannelPage: 3,
	}

	legalizedInternals, legalizedExternals := legalizedReferrals(internalReferrals, externalReferrals, playCount)

	assert.Equal(t, expectdInternals, legalizedInternals, "internals should aggregate non-whitelisted internal sources")
	assert.Equal(t, externalReferrals, legalizedExternals, "externals should be individualized")
}

func TestLegalizedReferrals_AllReferralsRevealed(t *testing.T) {
	var playCount int64 = 20

	internalReferrals := map[string]int64{
		"front_page_featured": 1,
		"my channel":          1,
		"another channel":     2,
	}

	externalReferrals := map[string]int64{
		"google":  3,
		"twitter": 3,
	}

	legalizedInternals, legalizedExternals := legalizedReferrals(internalReferrals, externalReferrals, playCount)

	assert.Equal(t, internalReferrals, legalizedInternals, "internals should be individualized")
	assert.Equal(t, externalReferrals, legalizedExternals, "externals should be individualized")
}
