package clients

import (
	"code.justin.tv/cb/dashy/config"
	"code.justin.tv/cb/dashy/internal/clients/dashy"
	"code.justin.tv/cb/dashy/internal/clients/dynamodb"
	"code.justin.tv/cb/dashy/internal/clients/sns"
	"code.justin.tv/cb/dashy/internal/clients/users"
	"code.justin.tv/cb/dashy/internal/clients/zephyr"
	"code.justin.tv/cb/dashy/internal/statsd"
)

// Clients contains all client interfaces to external resources
// required by the application.
type Clients struct {
	DynamoDB *DynamoDB
	Zephyr   zephyr.Zephyr
	SNS      sns.SNS
	Users    users.Service
	Statter  statsd.Client
}

// DynamoDB contains the interfaces to various AWS DynamoDB tables.
// Each client interface holds a connection pool to the respective
// table.
type DynamoDB struct {
	Tips                   dynamodb.TipsAdapter
	TagFilterStat          dynamodb.TagFilterStatAdapter
	NotificationsAnalytics dynamodb.NotificationsAnalyticsAdapter
}

// NewClients instantiates all necessary clients and returns
// a wrapper Clients struct.
func NewClients() (*Clients, error) {
	env := config.Environment
	region := "us-west-2"

	statsDClient := statsd.NewClient(env)

	zephyrDDB, err := zephyr.NewDynamoDB(&zephyr.ClientParams{
		Env:    env,
		Region: region,
	})
	if err != nil {
		return nil, err
	}

	dashyDDB, err := dashy.NewDynamoDB(&dashy.ClientParams{
		Env:    env,
		Region: region,
	})

	if err != nil {
		return nil, err
	}

	dynamoDBTipsAdapter, err := dynamodb.NewTipsAdapter(dashyDDB)
	if err != nil {
		return nil, err
	}

	dynamoTagFilterStatAdapter, err := dynamodb.NewTagFilterStatAdapter(zephyrDDB)
	if err != nil {
		return nil, err
	}

	dynamoNotificationsAnalyticsAdapter, err := dynamodb.NewNotificationsAnalyticsAdapter(zephyrDDB)
	if err != nil {
		return nil, err
	}

	snsClient, err := sns.NewClient(env, region, config.Values.SNSDashboardAccessTopic)
	if err != nil {
		return nil, err
	}

	usersClient, err := users.NewClient(config.Values.UsersServiceHost, statsDClient.Statter)
	if err != nil {
		return nil, err
	}

	zephyrClient, err := zephyr.NewClient(&zephyr.ClientParams{
		Env:    env,
		Region: region,
		Users:  usersClient,
	})
	if err != nil {
		return nil, err
	}

	return &Clients{
		DynamoDB: &DynamoDB{
			Tips:                   dynamoDBTipsAdapter,
			TagFilterStat:          dynamoTagFilterStatAdapter,
			NotificationsAnalytics: dynamoNotificationsAnalyticsAdapter,
		},
		Zephyr:  zephyrClient,
		SNS:     snsClient,
		Statter: statsDClient,
		Users:   usersClient,
	}, nil
}
