package dynamodb

import (
	"context"
	"fmt"
	"net/http"
	"strconv"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials/ec2rolecreds"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/endpoints"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"
	"github.com/aws/aws-sdk-go/service/sts"
	"github.com/pkg/errors"
)

type tipsAdapter struct {
	client dynamodbiface.DynamoDBAPI
}

type tagFilterStatAdapter struct {
	client dynamodbiface.DynamoDBAPI
}

type notificationsAnalyticsAdapter struct {
	client dynamodbiface.DynamoDBAPI
}

// NewDynamoDBFromProfile creates a DynamoDB client when developing locally and
// using isengard_credentials (or any other process) with the
// credentials_process option.
func NewDynamoDBFromProfile(profile string, region string) (dynamodbiface.DynamoDBAPI, error) {
	sess, err := session.NewSessionWithOptions(session.Options{
		Profile: profile,
		Config: aws.Config{
			Region: aws.String(region),
		},
	})

	if err != nil {
		return nil, err
	}

	return dynamodb.New(sess), nil
}

// Allows us to match and change endpoints used by the session to access AWS resources.
// Added to comply with STS regional endpoint call required by Amazon.
func customEndpointResolverFunc(service, region string, optFns ...func(*endpoints.Options)) (endpoints.ResolvedEndpoint, error) {
	if service == sts.EndpointsID {
		return endpoints.ResolvedEndpoint{
			URL:           fmt.Sprintf("https://sts.%s.amazonaws.com", region),
			SigningRegion: region,
		}, nil
	}
	return endpoints.DefaultResolver().EndpointFor(service, region, optFns...)
}

// NewDynamoDBFromIAMRole creates a DynamoDB client using the IAM Role that is
// provided. This function is useful when you need to access DynamoDB in another
// account.
func NewDynamoDBFromIAMRole(iamRole string, region string) (dynamodbiface.DynamoDBAPI, error) {
	sess, err := session.NewSession(&aws.Config{
		Region: aws.String(region),
	})
	if err != nil {
		return nil, err
	}

	creds := ec2rolecreds.NewCredentials(sess)

	sess, err = session.NewSession(&aws.Config{
		Credentials: creds,
		Region:      aws.String(region),
		// Our version of AWS SDK is v1.17.2 and the minimum aws.Config STS region flag is available in v1.25.18 of the SDK
		// TODO: When we upgrade to a STSRegionalEndpoint compatible AWS SDK version, simplify this by using that param instead of EndpointResolver
		// Link: https://w.amazon.com/bin/view/AWSAuth/STS/RegionalOnboarding#Go
		EndpointResolver: endpoints.ResolverFunc(customEndpointResolverFunc),
	})

	if err != nil {
		return nil, err
	}
	iamCreds := stscreds.NewCredentials(sess, iamRole)

	return dynamodb.New(sess, &aws.Config{
		Credentials: iamCreds,
		HTTPClient: &http.Client{
			Timeout: 8 * time.Second,
			Transport: &http.Transport{
				MaxIdleConnsPerHost: 100,
			},
		},
		MaxRetries: aws.Int(3),
		Region:     aws.String(region),
	}), nil
}

// NewDynamoDBFromEC2Credentials creates a DynamoDB client using the role
// associated with the EC2 instance running the code.
func NewDynamoDBFromEC2Credentials(region string) (dynamodbiface.DynamoDBAPI, error) {
	sess, err := session.NewSession(&aws.Config{
		Region: aws.String(region),
	})
	if err != nil {
		return nil, err
	}

	creds := ec2rolecreds.NewCredentials(sess)

	sess, err = session.NewSession(&aws.Config{
		Credentials: creds,
		HTTPClient: &http.Client{
			Timeout: 5 * time.Second,
			Transport: &http.Transport{
				MaxIdleConnsPerHost: 100,
			},
		},
		Region: aws.String(region),
	})

	if err != nil {
		return nil, err
	}

	return dynamodb.New(sess), nil
}

// NewTipsAdapter create new processor.
func NewTipsAdapter(client dynamodbiface.DynamoDBAPI) (TipsAdapter, error) {
	return &tipsAdapter{
		client: client,
	}, nil
}

func NewTagFilterStatAdapter(client dynamodbiface.DynamoDBAPI) (TagFilterStatAdapter, error) {
	return &tagFilterStatAdapter{
		client: client,
	}, nil
}

func NewNotificationsAnalyticsAdapter(client dynamodbiface.DynamoDBAPI) (NotificationsAnalyticsAdapter, error) {
	return &notificationsAnalyticsAdapter{
		client: client,
	}, nil
}

func (c *tipsAdapter) GetAllByKey(ctx context.Context, language string) ([]Tip, error) {
	keyCondition := aws.String("Lang = :language")
	conditionAttrValues := map[string]*dynamodb.AttributeValue{
		":language": {
			S: aws.String(language),
		},
	}

	output, err := c.client.QueryWithContext(ctx, &dynamodb.QueryInput{
		TableName:                 aws.String(TableStreamSummaryTips),
		KeyConditionExpression:    keyCondition,
		ExpressionAttributeValues: conditionAttrValues,
	})

	if err != nil {
		return nil, err
	}

	if *output.Count == 0 {
		return nil, nil
	}

	result := make([]Tip, *output.Count)
	for ind, value := range output.Items {

		var body, url string

		if value["Body"] != nil {
			body = *value["Body"].S
		}
		if value["Url"] != nil {
			url = *value["Url"].S
		}

		result[ind] = Tip{
			Body: body,
			URL:  url,
		}
	}

	return result, nil
}

func (c *tagFilterStatAdapter) GetByChannelID(ctx context.Context, channelID int64) ([]TagFilterStat, error) {
	output, err := c.client.GetItemWithContext(ctx, &dynamodb.GetItemInput{
		TableName: aws.String(TableTagFilters),
		Key: map[string]*dynamodb.AttributeValue{
			"ChannelID": {
				N: aws.String(strconv.FormatInt(channelID, 10)),
			},
		},
	})

	if err != nil {
		return nil, err
	}

	if output.Item == nil {
		return make([]TagFilterStat, 0), nil
	}

	item := output.Item

	if item["data"] == nil {
		return make([]TagFilterStat, 0), nil
	}

	tags := make([]TagFilterStat, len(item["data"].L))

	for idx, tag := range item["data"].L {
		percentage, err := strconv.ParseFloat(*(tag.M["percentage"].N), 64)
		if err != nil {
			percentage = 0
		}
		tags[idx] = TagFilterStat{
			TagID:      *(tag.M["tagID"].S),
			Percentage: percentage,
		}
	}

	return tags, nil
}

func (c *notificationsAnalyticsAdapter) GetAllByChannelIDAndTime(ctx context.Context, channelID string, startTime, endTime time.Time) ([]NotificationsAnalytics, error) {
	tableName := TableNotificationsAnalytics

	var exclusiveStartKey map[string]*dynamodb.AttributeValue
	keyCondition := aws.String("channel_id = :channelID AND #T BETWEEN :startTime AND :endTime")
	attributeValues := map[string]*dynamodb.AttributeValue{
		":channelID": {
			S: aws.String(channelID),
		},
		":startTime": {
			N: aws.String(fmt.Sprintf("%d", startTime.Unix())),
		},
		":endTime": {
			N: aws.String(fmt.Sprintf("%d", endTime.Unix())),
		},
	}
	attributeNames := map[string]*string{
		"#T": aws.String("stream_start_time"),
	}

	analytics := []NotificationsAnalytics{}

	for {
		output, err := c.client.QueryWithContext(ctx, &dynamodb.QueryInput{
			TableName:                 aws.String(tableName),
			ScanIndexForward:          aws.Bool(true),
			KeyConditionExpression:    keyCondition,
			ExpressionAttributeValues: attributeValues,
			ExpressionAttributeNames:  attributeNames,
			ExclusiveStartKey:         exclusiveStartKey,
		})

		if err != nil {
			return nil, errors.Wrapf(err, "failed to query %s", tableName)
		}

		for _, item := range output.Items {
			var analytic NotificationsAnalytics
			err := dynamodbattribute.UnmarshalMap(item, &analytic)
			if err != nil {
				return nil, errors.Wrap(err, "failed to build analytic")
			}

			analytics = append(analytics, analytic)
		}

		if output.LastEvaluatedKey == nil {
			break
		}

		exclusiveStartKey = output.LastEvaluatedKey
	}

	return analytics, nil
}
