// +build integration

package dynamodb_test

import (
	"context"
	"math/rand"
	"testing"
	"time"

	"code.justin.tv/cb/dashy/config"
	"code.justin.tv/cb/dashy/internal/clients/dynamodb"
	"github.com/aws/aws-sdk-go/aws"
	ddb "github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"
	"github.com/satori/go.uuid"
	"github.com/stretchr/testify/suite"
)

type DynamodbTestSuite struct {
	suite.Suite

	ctx    context.Context
	cancel context.CancelFunc

	dynamodbClient                dynamodbiface.DynamoDBAPI
	notificationsAnalyticsAdapter dynamodb.NotificationsAnalyticsAdapter
}

func TestIntegration(t *testing.T) {
	suite.Run(t, new(DynamodbTestSuite))
}

func (s *DynamodbTestSuite) SetupSuite() {
	env := config.Environment
	region := "us-west-2"

	var err error
	s.dynamodbClient, err = dynamodb.NewDynamoClient(env, region)
	s.Require().NoError(err)

	s.notificationsAnalyticsAdapter, err = dynamodb.NewNotificationsAnalyticsAdapter(s.dynamodbClient)
	s.Require().NoError(err)
}

func (s *DynamodbTestSuite) SetupTest() {
	ctx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	s.ctx = ctx
	s.cancel = cancel
}

func (s *DynamodbTestSuite) TearDownTest() {
	s.cancel()
}

func (s *DynamodbTestSuite) TestGetAllByChannelIDAndTime() {
	// Create test data
	channelID := generateRandomTestString()
	now := time.Unix(time.Now().Unix(), 0) // Ensures second granularity
	startTimes := []time.Time{now, now.Add(24 * time.Hour)}
	inputAnalytics := map[string]dynamodb.NotificationsAnalytics{}

	for _, startTime := range startTimes {
		analytic := generateNotificationAnalytics(channelID, startTime)
		inputAnalytics[analytic.BroadcastID] = analytic
	}

	err := s.insertAnalytics(inputAnalytics)
	s.NoError(err)

	// There should be none yesterday
	outputAnalytics, err := s.notificationsAnalyticsAdapter.GetAllByChannelIDAndTime(s.ctx, channelID, now.Add(-2*time.Hour), now.Add(-1*time.Hour))
	s.Require().NoError(err)
	s.Equal(0, len(outputAnalytics))

	// There should be 1 today
	outputAnalytics, err = s.notificationsAnalyticsAdapter.GetAllByChannelIDAndTime(s.ctx, channelID, now.Add(-2*time.Hour), now.Add(1*time.Hour))
	s.Require().NoError(err)
	s.Equal(1, len(outputAnalytics))
	s.compareAnalytics(inputAnalytics, outputAnalytics)

	// There should be 1 today and 1 tomorrow
	outputAnalytics, err = s.notificationsAnalyticsAdapter.GetAllByChannelIDAndTime(s.ctx, channelID, now.Add(-2*time.Hour), now.Add(25*time.Hour))
	s.Require().NoError(err)
	s.Equal(2, len(outputAnalytics))
	s.compareAnalytics(inputAnalytics, outputAnalytics)
}

func (s *DynamodbTestSuite) compareAnalytics(inputAnalytics map[string]dynamodb.NotificationsAnalytics, outputAnalytics []dynamodb.NotificationsAnalytics) {
	for _, outputAnalytic := range outputAnalytics {
		inputAnalytic, ok := inputAnalytics[outputAnalytic.BroadcastID]
		s.True(ok)
		s.Equal(inputAnalytic, outputAnalytic)
	}
}

func (s *DynamodbTestSuite) insertAnalytics(analytics map[string]dynamodb.NotificationsAnalytics) error {
	for _, analytic := range analytics {
		item, err := dynamodbattribute.MarshalMap(analytic)
		if err != nil {
			return err
		}

		if _, err := s.dynamodbClient.PutItemWithContext(s.ctx, &ddb.PutItemInput{
			TableName: aws.String(dynamodb.TableNotificationsAnalytics),
			Item:      item,
		}); err != nil {
			return err
		}
	}

	return nil
}

func generateRandomTestString() string {
	return "test_" + uuid.NewV4().String()
}

func generateNotificationAnalytics(channelID string, startTime time.Time) dynamodb.NotificationsAnalytics {
	rand.Seed(startTime.Unix())
	return dynamodb.NotificationsAnalytics{
		ChannelID:          channelID,
		BroadcastID:        generateRandomTestString(),
		StartTime:          startTime,
		FollowerCount:      rand.Int63(),
		EngagementCount:    rand.Int63(),
		NotificationCount:  rand.Int63(),
		CustomNotification: generateRandomTestString(),
	}
}
