package sns

import (
	"context"
	"encoding/json"
	"net/http"
	"time"

	"code.justin.tv/cb/dashy/config"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/credentials/ec2rolecreds"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sns"
)

// sns is SQS client.
type snsClient struct {
	Client *sns.SNS
	Topic  string
}

// NewClient creates an instance of an sns client
func NewClient(env string, region string, topic string) (SNS, error) {
	creds, err := generateCredentials(env, region)
	if err != nil {
		return nil, err
	}

	sess, err := session.NewSession(&aws.Config{
		Credentials: creds,
		HTTPClient: &http.Client{
			Timeout: 5 * time.Second,
			Transport: &http.Transport{
				MaxIdleConnsPerHost: 100,
			},
		},
		MaxRetries: aws.Int(3),
		Region:     aws.String(region),
	})

	if err != nil {
		return nil, err
	}

	return &snsClient{
		Client: sns.New(sess),
		Topic:  topic,
	}, nil
}

// generateCredentials returns AWS Credentials object for the current env
func generateCredentials(env string, region string) (*credentials.Credentials, error) {
	if env == config.Development {
		return credentials.NewSharedCredentials("", "twitch-web-dev"), nil
	}

	sess, err := session.NewSession(&aws.Config{
		Region: aws.String(region),
	})
	if err != nil {
		return nil, err
	}

	return ec2rolecreds.NewCredentials(sess), nil
}

// SendToTopic sends a message to the specified topic
func (c *snsClient) SendToTopic(ctx context.Context, message DashboardAccessMessage) error {
	json, err := json.Marshal(message)
	if err != nil {
		return err
	}

	input := &sns.PublishInput{
		Message:  aws.String(string(json)),
		TopicArn: aws.String(c.Topic),
	}

	_, err = c.Client.PublishWithContext(ctx, input)
	return err
}
