package zephyr

import (
	"time"

	dynamo "code.justin.tv/cb/dashy/internal/clients/dynamodb"

	"code.justin.tv/cb/dashy/config"
	"code.justin.tv/cb/dashy/internal/clients/users"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"
)

const (
	day                = 24 * time.Hour
	dynamoDBTimeFormat = "2006-01-02 15:04:05"
)

// Start is the earliest date that zephyr stores data for
var Start = time.Date(2017, time.January, 1, 0, 0, 0, 0, time.UTC)

type Client struct {
	dynamo     dynamodbiface.DynamoDBAPI
	users      users.Service
	production bool
}

type ClientParams struct {
	Env    string
	Region string
	Users  users.Service
}

// DynamoSession is how zephyr represents a session in dynamo
type DynamoSession struct {
	SegmentID        string          `dynamodbav:"segment_id"`
	ChannelID        string          `dynamodbav:"channel_id"`
	SegmentStartTime DynamoTimestamp `dynamodbav:"segment_start_time"`
	SegmentEndTime   DynamoTimestamp `dynamodbav:"segment_end_time"`
	IsSegmented      bool            `dynamodbav:"is_segmented"`
}

// DynamoTimestamp is used to marshal/unmarshal timestamps from dynamo
type DynamoTimestamp struct {
	Converted time.Time
}

type dynamoVideoPlayReferrer struct {
	SegmentID         string           `dynamodbav:"segment_id"`
	InternalBreakdown map[string]int64 `dynamodbav:"internal_referrers"`
	ExternalBreakdown map[string]int64 `dynamodbav:"external_referrers"`
}

type dynamoVideoPlayPlatform struct {
	SegmentID         string           `dynamodbav:"segment_id"`
	PlatformBreakdown map[string]int64 `dynamodbav:"platform_breakdown"`
}

type dynamoVideoPlayCountry struct {
	SegmentID        string           `dynamodbav:"segment_id"`
	CountryBreakdown map[string]int64 `dynamodbav:"country_breakdown"`
}

type dynamoVideoPlayViews struct {
	SegmentID   string `dynamodbav:"segment_id"`
	UniqueViews int64  `dynamodbav:"unique_views,int"`
	TotalViews  int64  `dynamodbav:"total_views,int"`
}

// NewDynamoDB returns a DynamoDB client connected to Zephyr Tables
func NewDynamoDB(params *ClientParams) (dynamodbiface.DynamoDBAPI, error) {
	if params.Env == config.Development {
		return dynamo.NewDynamoDBFromProfile("twitch-ce-analytics-dev", params.Region)
	} else {
		return dynamo.NewDynamoDBFromIAMRole(config.Values.ZephyrIAMRole, params.Region)
	}
}

// NewClient creates a new adapter to access zephyr dynamo tables
func NewClient(params *ClientParams) (*Client, error) {
	zephyrDDB, err := NewDynamoDB(params)

	if err != nil {
		return nil, err
	}

	return &Client{
		dynamo:     zephyrDDB,
		production: true,
		users:      params.Users,
	}, nil
}

// MarshalDynamoDBAttributeValue is the custom marshaller for the DynamoTimestamp type
func (propTime *DynamoTimestamp) MarshalDynamoDBAttributeValue(av *dynamodb.AttributeValue) error {
	timeObj := propTime.Converted
	timeString := timeObj.Format(dynamoDBTimeFormat)
	av.S = &timeString

	return nil
}

// UnmarshalDynamoDBAttributeValue is the custom unmarshaller for the DynamoTimestamp type
func (propTime *DynamoTimestamp) UnmarshalDynamoDBAttributeValue(av *dynamodb.AttributeValue) error {
	timeString := av.S
	if timeString == nil {
		return nil
	}

	timeObj, err := time.Parse(dynamoDBTimeFormat, *timeString)
	if err != nil {
		return err
	}

	propTime.Converted = timeObj
	return nil
}
