package zephyr

import (
	"context"
	"strconv"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/pkg/errors"
)

// UniqueChatter is a single timestemp struct for a unique chatter event from zephyr
type UniqueChatter struct {
	ChannelID      int64
	UniqueChatters int64
	TotalMessages  int64
	Time           time.Time
}

// GetUniqueChattersByTime returns all new unique chatters by time series for
// a given channelID between startTime and endTime. Returns an array of
// UniqueChatters structs or an error if something went wrong. Assumes startTime
// and endTime are valid.
func (c *Client) GetUniqueChattersByTime(ctx context.Context, channelID int64, startTime time.Time, endTime time.Time) ([]UniqueChatter, error) {
	tableName := "ChatMessageTimeseries"

	var exclusiveStartKey map[string]*dynamodb.AttributeValue
	keyCondition := aws.String("channel_id = :channelID AND #T BETWEEN :startTime AND :endTime")
	attributeValues := map[string]*dynamodb.AttributeValue{
		":channelID": {
			S: aws.String(strconv.FormatInt(channelID, 10)),
		},
		":startTime": {
			S: aws.String(startTime.Format(dynamoDBTimeFormat)),
		},
		":endTime": {
			S: aws.String(endTime.Format(dynamoDBTimeFormat)),
		},
	}
	attributeNames := map[string]*string{
		"#T": aws.String("timestamp"),
	}

	uniqueChatters := []UniqueChatter{}
	for {
		output, err := c.dynamo.QueryWithContext(ctx, &dynamodb.QueryInput{
			TableName:                 aws.String(tableName),
			ScanIndexForward:          aws.Bool(true),
			KeyConditionExpression:    keyCondition,
			ExpressionAttributeValues: attributeValues,
			ExpressionAttributeNames:  attributeNames,
			ExclusiveStartKey:         exclusiveStartKey,
		})

		if err != nil {
			return nil, errors.Wrapf(err, "failed to query %s", tableName)
		}

		for _, item := range output.Items {
			uniqueChatter, err := c.buildUniqueChatters(item)
			if err != nil {
				return nil, errors.Wrap(err, "failed to build unique chatters")
			}

			uniqueChatters = append(uniqueChatters, *uniqueChatter)
		}

		if output.LastEvaluatedKey == nil {
			break
		}

		exclusiveStartKey = output.LastEvaluatedKey
	}

	return uniqueChatters, nil
}

func (c *Client) buildUniqueChatters(item map[string]*dynamodb.AttributeValue) (*UniqueChatter, error) {
	channelID, err := strconv.ParseInt(*item["channel_id"].S, 10, 64)
	if err != nil {
		return nil, err
	}

	timetime, err := time.Parse(dynamoDBTimeFormat, *item["timestamp"].S)
	if err != nil {
		return nil, err
	}

	var uniqueChatters, totalMessages int64
	if item["unique_chatters"] != nil {
		if parsed, err := strconv.ParseInt(*item["unique_chatters"].N, 10, 64); err == nil {
			uniqueChatters = parsed
		}
	}

	if item["total_messages"] != nil {
		if parsed, err := strconv.ParseInt(*item["total_messages"].N, 10, 64); err == nil {
			totalMessages = parsed
		}
	}

	return &UniqueChatter{
		ChannelID:      channelID,
		UniqueChatters: uniqueChatters,
		TotalMessages:  totalMessages,
		Time:           timetime,
	}, nil
}
