package zephyr

import (
	"context"
	"strconv"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/pkg/errors"
)

const (
	clipCreatesTimestampColumn = "timestamp"
)

// ClipCreate represents a create count
type ClipCreate struct {
	ChannelID int64
	Creates   int64
	Time      time.Time
}

// GetClipCreatesByTime returns all clip creates within a given time range
func (c *Client) GetClipCreatesByTime(ctx context.Context, channelID int64, startTime time.Time, endTime time.Time) ([]ClipCreate, error) {
	tableName := "VideoPlayClipsCreateTimeseries"

	var exclusiveStartKey map[string]*dynamodb.AttributeValue
	keyCondition := aws.String("channel_id = :channelID AND #T BETWEEN :startTime AND :endTime")
	attributeValues := map[string]*dynamodb.AttributeValue{
		":channelID": {
			S: aws.String(strconv.FormatInt(channelID, 10)),
		},
		":startTime": {
			S: aws.String(startTime.Format(dynamoDBTimeFormat)),
		},
		":endTime": {
			S: aws.String(endTime.Format(dynamoDBTimeFormat)),
		},
	}
	attributeNames := map[string]*string{
		"#T": aws.String(clipCreatesTimestampColumn),
	}

	clipCreates := []ClipCreate{}
	for {
		output, err := c.dynamo.QueryWithContext(ctx, &dynamodb.QueryInput{
			TableName:                 aws.String(tableName),
			ScanIndexForward:          aws.Bool(true),
			KeyConditionExpression:    keyCondition,
			ExpressionAttributeValues: attributeValues,
			ExpressionAttributeNames:  attributeNames,
			ExclusiveStartKey:         exclusiveStartKey,
		})

		if err != nil {
			return nil, errors.Wrapf(err, "failed to query %s", tableName)
		}

		for _, item := range output.Items {
			clipCreate, err := buildClipCreate(item)
			if err != nil {
				return nil, errors.Wrap(err, "failed to build clip create")
			}

			clipCreates = append(clipCreates, *clipCreate)
		}

		if output.LastEvaluatedKey == nil {
			break
		}

		exclusiveStartKey = output.LastEvaluatedKey
	}

	return clipCreates, nil
}

func buildClipCreate(item map[string]*dynamodb.AttributeValue) (*ClipCreate, error) {
	channelID, err := strconv.ParseInt(*item["channel_id"].S, 10, 64)
	if err != nil {
		return nil, err
	}

	timetime, err := time.Parse(dynamoDBTimeFormat, *item[clipCreatesTimestampColumn].S)
	if err != nil {
		return nil, err
	}

	var count int64
	if item["creates"] != nil {
		if parsed, err := strconv.ParseInt(*item["creates"].N, 10, 64); err == nil {
			count = parsed
		}
	}

	return &ClipCreate{
		ChannelID: channelID,
		Creates:   count,
		Time:      timetime,
	}, nil
}
