package zephyr

import (
	"context"
	"strconv"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/pkg/errors"
)

const (
	clipPlaysTimestampColumn = "timestamp"
	playsColumn              = "plays"
	referrersColumn          = "referrers"
)

// ClipPlay represents a count of clip plays and play referrer stats
type ClipPlay struct {
	ChannelID int64
	Plays     int64
	Referrers map[string]int64
	Time      time.Time
}

// GetClipPlaysByTime returns all clip plays and their demographic breakdown by time
func (c *Client) GetClipPlaysByTime(ctx context.Context, channelID int64, startTime time.Time, endTime time.Time) ([]ClipPlay, error) {
	tableName := "VideoPlayClipsReferrerTimeseries"

	var exclusiveStartKey map[string]*dynamodb.AttributeValue
	keyCondition := aws.String("channel_id = :channelID AND #T BETWEEN :startTime AND :endTime")
	attributeValues := map[string]*dynamodb.AttributeValue{
		":channelID": {
			S: aws.String(strconv.FormatInt(channelID, 10)),
		},
		":startTime": {
			S: aws.String(startTime.Format(dynamoDBTimeFormat)),
		},
		":endTime": {
			S: aws.String(endTime.Format(dynamoDBTimeFormat)),
		},
	}
	attributeNames := map[string]*string{
		"#T": aws.String(clipPlaysTimestampColumn),
	}

	clipPlays := []ClipPlay{}
	for {
		output, err := c.dynamo.QueryWithContext(ctx, &dynamodb.QueryInput{
			TableName:                 aws.String(tableName),
			ScanIndexForward:          aws.Bool(true),
			KeyConditionExpression:    keyCondition,
			ExpressionAttributeValues: attributeValues,
			ExpressionAttributeNames:  attributeNames,
			ExclusiveStartKey:         exclusiveStartKey,
		})

		if err != nil {
			return nil, errors.Wrapf(err, "failed to query %s", tableName)
		}

		for _, item := range output.Items {
			clipPlay, err := buildClipPlay(item)
			if err != nil {
				return nil, errors.Wrap(err, "failed to build clip play")
			}

			clipPlays = append(clipPlays, *clipPlay)
		}

		if output.LastEvaluatedKey == nil {
			break
		}

		exclusiveStartKey = output.LastEvaluatedKey
	}

	return clipPlays, nil
}

func buildClipPlay(item map[string]*dynamodb.AttributeValue) (*ClipPlay, error) {
	channelID, err := strconv.ParseInt(*item["channel_id"].S, 10, 64)
	if err != nil {
		return nil, err
	}

	timetime, err := time.Parse(dynamoDBTimeFormat, *item[clipPlaysTimestampColumn].S)
	if err != nil {
		return nil, err
	}

	var count int64
	if item[playsColumn] != nil {
		if parsed, err := strconv.ParseInt(*item[playsColumn].N, 10, 64); err == nil {
			count = parsed
		}
	}

	var playBreakdown map[string]int64
	if item[referrersColumn] != nil {
		err := dynamodbattribute.Unmarshal(item[referrersColumn], &playBreakdown)
		if err != nil {
			return nil, err
		}
	}

	return &ClipPlay{
		ChannelID: channelID,
		Plays:     count,
		Referrers: playBreakdown,
		Time:      timetime,
	}, nil
}
