package zephyr

import (
	"context"
	"strconv"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/pkg/errors"
)

// Commercial is a single timestemp struct for a commercial chatter event from zephyr
type Commercial struct {
	ChannelID        int64
	CommercialLength int64
	CommercialCount  int64
	Time             time.Time
}

// GetCommercialByTime returns all the commercial length and count by time series for
// a given channelID between startTime and endTime. Returns an array of
// GetCommercialByTime structs or an error if something went wrong. Assumes startTime
// and endTime are valid.
func (c *Client) GetCommercialByTime(ctx context.Context, channelID int64, startTime time.Time, endTime time.Time) ([]Commercial, error) {
	tableName := "CommercialTimeseries"

	var exclusiveStartKey map[string]*dynamodb.AttributeValue
	keyCondition := aws.String("channel_id = :channelID AND #T BETWEEN :startTime AND :endTime")
	attributeValues := map[string]*dynamodb.AttributeValue{
		":channelID": {
			S: aws.String(strconv.FormatInt(channelID, 10)),
		},
		":startTime": {
			S: aws.String(startTime.Format(dynamoDBTimeFormat)),
		},
		":endTime": {
			S: aws.String(endTime.Format(dynamoDBTimeFormat)),
		},
	}
	attributeNames := map[string]*string{
		"#T": aws.String("timestamp"),
	}

	commercials := []Commercial{}
	for {
		output, err := c.dynamo.QueryWithContext(ctx, &dynamodb.QueryInput{
			TableName:                 aws.String(tableName),
			ScanIndexForward:          aws.Bool(true),
			KeyConditionExpression:    keyCondition,
			ExpressionAttributeValues: attributeValues,
			ExpressionAttributeNames:  attributeNames,
			ExclusiveStartKey:         exclusiveStartKey,
		})

		if err != nil {
			return nil, errors.Wrapf(err, "failed to query %s", tableName)
		}

		for _, item := range output.Items {
			commercial, err := c.buildCommercial(item)
			if err != nil {
				return nil, errors.Wrap(err, "failed to build commercial")
			}

			commercials = append(commercials, *commercial)
		}

		if output.LastEvaluatedKey == nil {
			break
		}

		exclusiveStartKey = output.LastEvaluatedKey
	}

	return commercials, nil
}

func (c *Client) buildCommercial(item map[string]*dynamodb.AttributeValue) (*Commercial, error) {
	channelID, err := strconv.ParseInt(*item["channel_id"].S, 10, 64)
	if err != nil {
		return nil, err
	}

	timetime, err := time.Parse(dynamoDBTimeFormat, *item["timestamp"].S)
	if err != nil {
		return nil, err
	}

	var commercialLength, commercialCount int64
	if item["commercial_length"] != nil {
		if parsed, err := strconv.ParseInt(*item["commercial_length"].N, 10, 64); err == nil {
			commercialLength = parsed
		}
	}

	if item["commercial_count"] != nil {
		if parsed, err := strconv.ParseInt(*item["commercial_count"].N, 10, 64); err == nil {
			commercialCount = parsed
		}
	}

	return &Commercial{
		ChannelID:        channelID,
		CommercialLength: commercialLength,
		CommercialCount:  commercialCount,
		Time:             timetime,
	}, nil
}
