package zephyr

import (
	"context"
	"strconv"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/pkg/errors"
)

// ConcurrentViewership contains the average concurrent viewer count for a given channel at a given timestamp.
type ConcurrentViewership struct {
	ChannelID    int64
	Timestamp    time.Time
	AverageCount float64
}

// GetConcurrentViewersByTime returns all new unique chatters by time series for
// a given channelID between startTime and endTime. Returns an array of
// UniqueChatters structs or an error if something went wrong. Assumes startTime
// and endTime are valid.
func (c *Client) GetConcurrentViewersByTime(ctx context.Context, channelID int64, startTime, endTime time.Time) ([]ConcurrentViewership, error) {
	tableName := "ConcurrentsTimeseries"

	var exclusiveStartKey map[string]*dynamodb.AttributeValue
	keyCondition := aws.String("channel_id = :channelID AND #T BETWEEN :startTime AND :endTime")
	attributeValues := map[string]*dynamodb.AttributeValue{
		":channelID": {
			S: aws.String(strconv.FormatInt(channelID, 10)),
		},
		":startTime": {
			S: aws.String(startTime.Format(dynamoDBTimeFormat)),
		},
		":endTime": {
			S: aws.String(endTime.Format(dynamoDBTimeFormat)),
		},
	}
	attributeNames := map[string]*string{
		"#T": aws.String("timestamp"),
	}

	concurrents := []ConcurrentViewership{}

	for {
		output, err := c.dynamo.QueryWithContext(ctx, &dynamodb.QueryInput{
			TableName:                 aws.String(tableName),
			ScanIndexForward:          aws.Bool(true),
			KeyConditionExpression:    keyCondition,
			ExpressionAttributeValues: attributeValues,
			ExpressionAttributeNames:  attributeNames,
			ExclusiveStartKey:         exclusiveStartKey,
		})

		if err != nil {
			return nil, errors.Wrapf(err, "failed to query %s", tableName)
		}

		for _, item := range output.Items {
			concurrent, err := c.buildConcurrentViewership(item)
			if err != nil {
				return nil, errors.Wrap(err, "failed to build unique chatters")
			}

			concurrents = append(concurrents, concurrent)
		}

		if output.LastEvaluatedKey == nil {
			break
		}

		exclusiveStartKey = output.LastEvaluatedKey
	}

	return concurrents, nil
}

func (c *Client) buildConcurrentViewership(item map[string]*dynamodb.AttributeValue) (ConcurrentViewership, error) {
	channelID, err := strconv.ParseInt(*item["channel_id"].S, 10, 64)
	if err != nil {
		return ConcurrentViewership{}, err
	}

	timestamp, err := time.Parse(dynamoDBTimeFormat, *item["timestamp"].S)
	if err != nil {
		return ConcurrentViewership{}, err
	}

	var average float64
	if item["average_concurrents"] != nil {
		if float, err := strconv.ParseFloat(*item["average_concurrents"].N, 64); err == nil {
			average = float
		}
	}

	return ConcurrentViewership{
		ChannelID:    channelID,
		Timestamp:    timestamp,
		AverageCount: average,
	}, nil
}
