package zephyr

import (
	"context"
	"strconv"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/pkg/errors"
)

// NewFollower is a single timestemp struct for a new follower event from zephyr
type NewFollower struct {
	ChannelID int64
	Count     int64
	Time      time.Time
}

// GetFollowersByTime returns all new incremental followers by time series for
// a given channelID between startTime and endTime. Returns an array of
// NewFollower structs or an error if something went wrong. Assumes startTime
// and endTime are valid.
func (c *Client) GetFollowersByTime(ctx context.Context, channelID int64, startTime time.Time, endTime time.Time) ([]NewFollower, error) {
	tableName := "ServerFollowTimeseries"

	var exclusiveStartKey map[string]*dynamodb.AttributeValue
	keyCondition := aws.String("channel_id = :channelID AND #T BETWEEN :startTime AND :endTime")
	attributeValues := map[string]*dynamodb.AttributeValue{
		":channelID": {
			S: aws.String(strconv.FormatInt(channelID, 10)),
		},
		":startTime": {
			S: aws.String(startTime.Format(dynamoDBTimeFormat)),
		},
		":endTime": {
			S: aws.String(endTime.Format(dynamoDBTimeFormat)),
		},
	}
	attributeNames := map[string]*string{
		"#T": aws.String("timestamp"),
	}

	newFollowers := []NewFollower{}
	for {
		output, err := c.dynamo.QueryWithContext(ctx, &dynamodb.QueryInput{
			TableName:                 aws.String(tableName),
			ScanIndexForward:          aws.Bool(true),
			KeyConditionExpression:    keyCondition,
			ExpressionAttributeValues: attributeValues,
			ExpressionAttributeNames:  attributeNames,
			ExclusiveStartKey:         exclusiveStartKey,
		})

		if err != nil {
			return nil, errors.Wrapf(err, "failed to query %s", tableName)
		}

		for _, item := range output.Items {
			newFollower, err := c.buildNewFollower(item)
			if err != nil {
				return nil, errors.Wrap(err, "failed to build new follower")
			}

			newFollowers = append(newFollowers, *newFollower)
		}

		if output.LastEvaluatedKey == nil {
			break
		}

		exclusiveStartKey = output.LastEvaluatedKey
	}

	return newFollowers, nil
}

func (c *Client) buildNewFollower(item map[string]*dynamodb.AttributeValue) (*NewFollower, error) {
	channelID, err := strconv.ParseInt(*item["channel_id"].S, 10, 64)
	if err != nil {
		return nil, err
	}

	timetime, err := time.Parse(dynamoDBTimeFormat, *item["timestamp"].S)
	if err != nil {
		return nil, err
	}

	var count int64
	if item["new_followers"] != nil {
		if parsed, err := strconv.ParseInt(*item["new_followers"].N, 10, 64); err == nil {
			count = parsed
		}
	}

	return &NewFollower{
		ChannelID: channelID,
		Count:     count,
		Time:      timetime,
	}, nil
}
