package zephyr

import (
	"context"
	"strconv"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/pkg/errors"
)

// Raid is a single timestemp struct for a unique raid event from zephyr
type Raid struct {
	ChannelID   int64
	ViewerCount int64
	Count       int64
	RaidID      string
	Time        time.Time
}

// GetRaidsByTime returns all  raids by time series for
// a given channelID between startTime and endTime. Returns an array of
// Raid structs or an error if something went wrong. Assumes startTime
// and endTime are valid.
func (c *Client) GetRaidsByTime(ctx context.Context, channelID int64, startTime time.Time, endTime time.Time) ([]Raid, error) {
	tableName := "RaidExecuteTimeseries"

	var exclusiveStartKey map[string]*dynamodb.AttributeValue
	keyCondition := aws.String("target_id = :targetID AND #T BETWEEN :startTime AND :endTime")
	attributeValues := map[string]*dynamodb.AttributeValue{
		":targetID": {
			S: aws.String(strconv.FormatInt(channelID, 10)),
		},
		":startTime": {
			S: aws.String(startTime.Format(dynamoDBTimeFormat)),
		},
		":endTime": {
			S: aws.String(endTime.Format(dynamoDBTimeFormat)),
		},
	}
	attributeNames := map[string]*string{
		"#T": aws.String("timestamp"),
	}

	raids := []Raid{}
	for {
		output, err := c.dynamo.QueryWithContext(ctx, &dynamodb.QueryInput{
			TableName:                 aws.String(tableName),
			ScanIndexForward:          aws.Bool(true),
			KeyConditionExpression:    keyCondition,
			ExpressionAttributeValues: attributeValues,
			ExpressionAttributeNames:  attributeNames,
			ExclusiveStartKey:         exclusiveStartKey,
		})

		if err != nil {
			return nil, errors.Wrapf(err, "failed to query %s", tableName)
		}

		for _, item := range output.Items {
			raid, err := c.buildRaid(item)
			if err != nil {
				return nil, errors.Wrap(err, "failed to build raid")
			}

			raids = append(raids, *raid)
		}

		if output.LastEvaluatedKey == nil {
			break
		}

		exclusiveStartKey = output.LastEvaluatedKey
	}

	return raids, nil
}

func (c *Client) buildRaid(item map[string]*dynamodb.AttributeValue) (*Raid, error) {
	channelID, err := strconv.ParseInt(*item["target_id"].S, 10, 64)
	if err != nil {
		return nil, err
	}

	timetime, err := time.Parse(dynamoDBTimeFormat, *item["timestamp"].S)
	if err != nil {
		return nil, err
	}

	var viewerCount, count int64
	var raidID string
	if item["viewer_count"] != nil {
		if parsed, err := strconv.ParseInt(*item["viewer_count"].N, 10, 64); err == nil {
			viewerCount = parsed
		}
	}

	if item["number_of_raids"] != nil {
		if parsed, err := strconv.ParseInt(*item["number_of_raids"].N, 10, 64); err == nil {
			count = parsed
		}
	}

	if item["raid_id"] != nil {
		raidID = *item["raid_id"].S
	}

	return &Raid{
		ChannelID:   channelID,
		ViewerCount: viewerCount,
		Count:       count,
		RaidID:      raidID,
		Time:        timetime,
	}, nil
}
