package zephyr

import (
	"context"
	"fmt"
	"sort"
	"strconv"
	"strings"
	"sync"
	"time"

	"code.justin.tv/cb/dashy/internal/legal"
	"code.justin.tv/cb/dashy/view/summary"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
	"golang.org/x/sync/errgroup"
)

const (
	zephyrSessionsTable                            = "Sessions"
	zephyrBroadcastIDSession                       = "BroadcastIDsSession"
	zephyrChannelConcurrentsSessionTable           = "ConcurrentsPerSession"
	zephyrCommercialSessionTable                   = "CommercialPerSession"
	zephyrRaidExecuteSessionTable                  = "RaidExecutePerSession"
	zephyrServerChatMessageSessionTable            = "ChatMessageSession"
	zephyrServerFollowSessionTable                 = "ServerFollowPerSession"
	zephyrSubscriptionsPurchaseSuccessSessionTable = "SubscriptionsPurchaseSuccessPerSession"
	zephyrVideoPlayGeographicalSessionTable        = "VideoPlayGeoPerSession"
	zephyrVideoPlayPlatformSessionTable            = "VideoPlayPlatformPerSession"
	zephyrVideoPlayReferrerSessionTable            = "VideoPlayReferrerPerSession"
	zephyrVideoPlayUniqueSessionTable              = "VideoPlayUniquePerSession"
	zephyrVideoPlayClipPlaysSessionTable           = "VideoPlayClipsReferrerPerSession"
	zephyrVideoPlayClipCreatesSessionTable         = "VideoPlayClipsCreatePerSession"
	zephyrVideoPlayHostRaidsSessionTable           = "MinuteWatchedHostedAndRaidedPerSession"

	singleAdLength = 30

	platformAndroid   = "android"
	platformIOS       = "ios"
	platformMobileWeb = "mobile_web"
	platformAggregate = "mobile_aggregated"

	referrerInternalOtherChannelPage = "other_channel_page"
	referrerInternalAggregate        = "twitch_aggregate"
	referrerExternalAggregate        = "external_aggregate"
)

var internalReferrerWhitelist = map[string]bool{
	"front_page_featured":      true,
	"creative_page_featured":   true,
	"hosted":                   true,
	"homepage":                 true,
	"email_live_notification":  true,
	"onsite_notification":      true,
	"followed_channel":         true,
	"directory_browse":         true,
	"top_nav_bar":              true,
	"recommended_channel":      true,
	"search":                   true,
	"clips_live":               true,
	"friend_presence":          true,
	"homepage_carousel":        true,
	"homepage_recommendations": true,
	"other":                    true,
}

var bucketedExternalReferrers = []string{
	"google",
	"youtube",
	"facebook",
	"t.co",
	"twitter",
	"reddit",
}

// lockingMap allows use to write to a map concurrently by locking it first and unlocking it when we finish.
// There are two main reasons to use this over sync.Map:
//
// 1. We lose all type safety when using interface{} as the type for keys and values.
// 2. sync.Map is not serialized to a json response. This means we would need to convert sync.Map to a regular
//    map ourselves, which would involve converting interface{} to string types manually, and handling those errors.
//
// This struct contains two inner maps, one for our summary and one for detailed breakdown stats. It is the intention
// that both these maps are then used to set the values in the summary.Session struct after all stats have been retrieved.
// Only a store method is provided for this type, since we don't actually perform any reading or deleting on this object.
type lockingMap struct {
	sync.Mutex
	InnerSummaryMap map[string]*string
	InnerStatsMap   map[string][]*summary.Stat
}

// Write a value to the summary mapping
func (l *lockingMap) storeSummary(key string, value *string) {
	l.Lock()
	l.InnerSummaryMap[key] = value
	l.Unlock()
}

// Write a value to the stats mapping
func (l *lockingMap) storeStats(key string, value []*summary.Stat) {
	l.Lock()
	l.InnerStatsMap[key] = value
	l.Unlock()
}

func (c *Client) GetSessionsSummaryByTime(ctx context.Context, channelID int64, startTime time.Time, endTime time.Time) ([]*summary.Session, error) {
	sessionTableName := zephyrSessionsTable

	keyCondition := aws.String("channel_id = :channelID AND segment_start_time BETWEEN :startTime AND :endTime")
	expressionAttributeValues := map[string]*dynamodb.AttributeValue{
		":channelID": {
			S: aws.String(strconv.FormatInt(channelID, 10)),
		},
		":startTime": {
			S: aws.String(startTime.Format(dynamoDBTimeFormat)),
		},
		":endTime": {
			S: aws.String(endTime.Format(dynamoDBTimeFormat)),
		},
	}

	output, err := c.dynamo.QueryWithContext(ctx, &dynamodb.QueryInput{
		TableName:                 aws.String(sessionTableName),
		ScanIndexForward:          aws.Bool(true),
		KeyConditionExpression:    keyCondition,
		ExpressionAttributeValues: expressionAttributeValues,
	})
	if err != nil {
		return nil, errors.Wrapf(err, "failed to query %s", sessionTableName)
	}

	if output == nil {
		log.WithFields(log.Fields{
			"channelID": channelID,
			"startTime": startTime.Format(time.RFC3339),
			"endTime":   startTime.Format(time.RFC3339),
		}).Info("query for session returned nil")
		return nil, nil
	}

	sessions := make([]*summary.Session, 0, len(output.Items))
	group, ctx := errgroup.WithContext(ctx)

	// Fetch and cache the channel login once when later requested
	channelLogin := fetchOnceStr(func() (string, error) {
		user, err := c.users.GetUserByID(ctx, strconv.FormatInt(channelID, 10))
		if err != nil {
			return "", errors.Wrap(err, "failed to GetUserByID")
		}

		if user.Login == nil {
			return "", errors.Wrap(fmt.Errorf("missing user Login field"), "failed to user's login")
		}

		return *user.Login, nil
	})

	for _, item := range output.Items {
		newSession, err := c.buildSession(item)
		if err != nil {
			return nil, errors.Wrap(err, "failed to build session")
		}

		sessionLockingMap := &lockingMap{
			InnerSummaryMap: make(map[string]*string),
			InnerStatsMap:   make(map[string][]*summary.Stat),
		}

		// average_concurrent_viewer_count, max_concurrent_viewer_count, max_viewer_time
		group.Go(func() error {
			err = c.sessionConcurrentViewers(ctx, newSession, sessionLockingMap)
			if err != nil {
				return errors.Wrap(err, "failed to get session concurrent viewers")
			}

			return nil
		})

		// followers_change
		group.Go(func() error {
			err = c.sessionFollow(ctx, newSession, sessionLockingMap)
			if err != nil {
				return errors.Wrap(err, "failed to get session follow")
			}

			return nil
		})

		// chatters_unique, messages_total
		group.Go(func() error {
			err = c.sessionChat(ctx, newSession, sessionLockingMap)
			if err != nil {
				return errors.Wrap(err, "failed to get session chat")
			}

			return nil
		})

		// commercial_breaks, total_commercial_length, commercial_density
		group.Go(func() error {
			err = c.sessionCommercial(ctx, newSession, sessionLockingMap)
			if err != nil {
				return errors.Wrap(err, "failed to get session commercial")
			}

			return nil
		})

		// raids_incoming
		group.Go(func() error {
			err = c.sessionRaid(ctx, newSession, sessionLockingMap)
			if err != nil {
				return errors.Wrap(err, "failed to get session raid")
			}

			return nil
		})

		// broadcast_ids
		group.Go(func() error {
			err = c.sessionBroadcastIDs(ctx, newSession, sessionLockingMap)
			if err != nil {
				return errors.Wrap(err, "failed to get session broadcast ids")
			}

			return nil
		})

		// video_play_geographics
		group.Go(func() error {
			err = c.sessionVideoPlayGeographics(ctx, newSession, sessionLockingMap)
			if err != nil {
				return errors.Wrap(err, "failed to get session video play geographics")
			}

			return nil
		})

		// video_play_platforms
		group.Go(func() error {
			err = c.sessionVideoPlayPlatforms(ctx, newSession, sessionLockingMap)
			if err != nil {
				return errors.Wrap(err, "failed to get session video play platforms")
			}

			return nil
		})

		// video_play_referral
		group.Go(func() error {
			err = c.sessionVideoPlayReferrers(ctx, newSession, sessionLockingMap, channelLogin)
			if err != nil {
				return errors.Wrap(err, "failed to get session video play referrers")
			}

			return nil
		})

		// video_play_total, video_play_unqiue
		group.Go(func() error {
			err = c.sessionVideoPlayCounts(ctx, newSession, sessionLockingMap)
			if err != nil {
				return errors.Wrap(err, "failed to get session video play counts")
			}

			return nil
		})

		// new_subscriptions
		group.Go(func() error {
			err = c.sessionSubscriptions(ctx, newSession, sessionLockingMap)
			if err != nil {
				return errors.Wrap(err, "failed to get session subscriptions")
			}

			return nil
		})

		// clip creates
		group.Go(func() error {
			err = c.sessionClipCreates(ctx, newSession, sessionLockingMap)
			if err != nil {
				return errors.Wrap(err, "failed to get session clip creates")
			}

			return nil
		})

		// clip plays and referrers
		group.Go(func() error {
			err = c.sessionClipPlays(ctx, newSession, sessionLockingMap)
			if err != nil {
				return errors.Wrap(err, "failed to get session clip plays")
			}

			return nil
		})

		// host/raid percent
		group.Go(func() error {
			err = c.sessionHostRaidPercentage(ctx, newSession, sessionLockingMap)
			if err != nil {
				return errors.Wrap(err, "failed to get session host/raid percentage")
			}

			return nil
		})

		newSession.Summary = sessionLockingMap.InnerSummaryMap
		newSession.Stats = sessionLockingMap.InnerStatsMap
		sessions = append(sessions, newSession)
	}

	if err := group.Wait(); err != nil {
		return nil, err
	}

	return sessions, nil
}

// buildSession will take a dynamodb record and parse the channel_id,
// segment_start_time, segment_end_time, segment_id, and is_segmented into a
// summary.Session struct
func (c *Client) buildSession(item map[string]*dynamodb.AttributeValue) (*summary.Session, error) {
	channelID, err := strconv.ParseInt(*item["channel_id"].S, 10, 64)
	if err != nil {
		return nil, errors.Wrap(err, "failed to channel_id")
	}

	startTime, err := time.Parse(dynamoDBTimeFormat, *item["segment_start_time"].S)
	if err != nil {
		return nil, errors.Wrap(err, "failed to parse segment_start_time")
	}

	// set the segment end time here if it's not available (in progress streams)
	var endTime time.Time
	if item["segment_end_time"] == nil || item["segment_end_time"].S == nil || *item["segment_end_time"].S == "" {
		// Set the end time to be 30 min from the current timestamp
		streamEnd := time.Now().Add(-30 * time.Minute)

		if streamEnd.Before(startTime) {
			endTime = startTime.UTC()
		} else {
			endTime = streamEnd.UTC()
		}
	} else {
		endTime, err = time.Parse(dynamoDBTimeFormat, *item["segment_end_time"].S)
		if err != nil {
			return nil, errors.Wrap(err, "failed to parse segment_end_time")
		}
	}

	// is_segmented is null for sessions from the flink job. Default these to false
	isSegmented := false
	if item["is_segmented"] != nil && item["is_segmented"].BOOL != nil {
		isSegmented = *item["is_segmented"].BOOL
	}

	return &summary.Session{
		ChannelID:   channelID,
		StartTime:   &startTime,
		EndTime:     &endTime,
		SegmentID:   *item["segment_id"].S,
		IsSegmented: isSegmented,
		Summary:     map[string]*string{},
		Stats:       map[string][]*summary.Stat{},
	}, nil
}

// queryZephyrSessionTable is a helper function to query a target table and
// and return a pointer of the dynamodb query output
func (c *Client) queryZephyrSessionTable(ctx context.Context, sess *summary.Session, tableName string) (*dynamodb.QueryOutput, error) {
	keyCondition := aws.String("segment_id = :segmentID")
	expressionAttributeValues := map[string]*dynamodb.AttributeValue{
		":segmentID": {
			S: aws.String(sess.SegmentID),
		},
	}

	output, err := c.dynamo.QueryWithContext(ctx, &dynamodb.QueryInput{
		TableName:                 aws.String(tableName),
		ScanIndexForward:          aws.Bool(true),
		KeyConditionExpression:    keyCondition,
		ExpressionAttributeValues: expressionAttributeValues,
	})

	if err != nil {
		return nil, errors.Wrapf(err, "failed to query %s", tableName)
	}

	return output, nil
}

// sessionConcurrentViewers will take a session's SegmentID and then
// query zephyr's channel-concurrent-session table to get max and average
// concurrent viewers, and save it onto the session struct
func (c *Client) sessionConcurrentViewers(ctx context.Context, sess *summary.Session, sessionLockingMap *lockingMap) error {
	tableName := zephyrChannelConcurrentsSessionTable
	output, err := c.queryZephyrSessionTable(ctx, sess, tableName)
	if err != nil {
		return err
	}

	sessionLockingMap.storeSummary("concurrents_average", nil)
	sessionLockingMap.storeSummary("concurrents_max", nil)
	sessionLockingMap.storeSummary("concurrents_peak_time", nil)
	sessionLockingMap.storeSummary("minutes_watched_total", nil)

	if *output.Count == 0 {
		return nil
	}

	item := output.Items[0]

	if _, ok := item["average_concurrents"]; ok {
		sessionLockingMap.storeSummary("concurrents_average", item["average_concurrents"].N)
	}
	if _, ok := item["max_concurrents"]; ok {
		sessionLockingMap.storeSummary("concurrents_max", item["max_concurrents"].N)
	}
	if _, ok := item["peak_concurrents_time"]; ok {
		unformattedTime, err := time.Parse(dynamoDBTimeFormat, *item["peak_concurrents_time"].S)
		if err != nil {
			log.WithError(err).Warn("failed to parse peak_concurrents_time from channel-concurrents-session")
		} else {
			formattedTime := unformattedTime.Format(time.RFC3339)
			sessionLockingMap.storeSummary("concurrents_peak_time", &formattedTime)
		}
	}
	if _, ok := item["total_minutes_watched"]; ok {
		sessionLockingMap.storeSummary("minutes_watched_total", item["total_minutes_watched"].N)
	}

	return nil
}

// sessionFollow will take a session's SegmentID and the query
// zephyr's server-follow-session table to get new followers, and save it onto
// the session struct
//
// TODO: missing total follower count
func (c *Client) sessionFollow(ctx context.Context, sess *summary.Session, sessionLockingMap *lockingMap) error {
	tableName := zephyrServerFollowSessionTable
	output, err := c.queryZephyrSessionTable(ctx, sess, tableName)
	if err != nil {
		return err
	}

	sessionLockingMap.storeSummary("followers_change", nil)

	if *output.Count == 0 {
		return nil
	}

	item := output.Items[0]
	if _, ok := item["new_followers"]; ok {
		sessionLockingMap.storeSummary("followers_change", item["new_followers"].N)
	}

	return nil
}

// sessionChat will take a session's SegmentID and then
// query zephyr's server-chat-message-session table to get unique chatters and
// total messages, and save it onto the session struct
func (c *Client) sessionChat(ctx context.Context, sess *summary.Session, sessionLockingMap *lockingMap) error {
	tableName := zephyrServerChatMessageSessionTable
	output, err := c.queryZephyrSessionTable(ctx, sess, tableName)
	if err != nil {
		return err
	}

	sessionLockingMap.storeSummary("chatters_unique", nil)
	sessionLockingMap.storeSummary("messages_total", nil)

	if *output.Count == 0 {
		return nil
	}

	item := output.Items[0]
	if _, ok := item["unique_chatters"]; ok {
		sessionLockingMap.storeSummary("chatters_unique", item["unique_chatters"].N)
	}
	if _, ok := item["total_messages"]; ok {
		sessionLockingMap.storeSummary("messages_total", item["total_messages"].N)
	}

	return nil
}

// sessionCommercial will take a session's SegmentID and the query
// zephyr's commercial-session table to get commercial length, and save it onto
// the session struct
func (c *Client) sessionCommercial(ctx context.Context, sess *summary.Session, sessionLockingMap *lockingMap) error {
	tableName := zephyrCommercialSessionTable
	output, err := c.queryZephyrSessionTable(ctx, sess, tableName)
	if err != nil {
		return err
	}

	sessionLockingMap.storeSummary("commercial_length_total", nil)
	sessionLockingMap.storeSummary("commercial_count", nil)
	sessionLockingMap.storeSummary("commercial_density", nil)

	if *output.Count == 0 {
		return nil
	}

	item := output.Items[0]
	var commercialLengthTotal float64
	if _, ok := item["commercial_length"]; ok {
		sessionLockingMap.storeSummary("commercial_length_total", item["commercial_length"].N)
		commercialLengthTotal, err = strconv.ParseFloat(*item["commercial_length"].N, 64)
		if err != nil {
			return errors.Wrap(err, "failed to parse commercial length")
		}
	}

	if _, ok := item["commercial_count"]; ok {
		sessionLockingMap.storeSummary("commercial_count", item["commercial_count"].N)
	}

	commercialDensity := (commercialLengthTotal / singleAdLength) / sess.EndTime.Sub(*sess.StartTime).Hours()
	commercialDensityString := strconv.FormatFloat(commercialDensity, 'f', -1, 64)
	sessionLockingMap.storeSummary("commercial_density", &commercialDensityString)

	return nil
}

// sessionRaid will take a session's SegmentID and the query
// zephyr's raid-execute-session table to get number_of_raids, and save it onto
// the session struct
func (c *Client) sessionRaid(ctx context.Context, sess *summary.Session, sessionLockingMap *lockingMap) error {
	tableName := zephyrRaidExecuteSessionTable
	output, err := c.queryZephyrSessionTable(ctx, sess, tableName)
	if err != nil {
		return err
	}

	sessionLockingMap.storeSummary("raids_incoming", nil)

	if *output.Count == 0 {
		return nil
	}

	item := output.Items[0]
	if _, ok := item["number_of_raids"]; ok {
		sessionLockingMap.storeSummary("raids_incoming", item["number_of_raids"].N)
	}

	return nil
}

// sessionBroadcastIDs will take a session's SegmentID and the query
// zephyr's broadcastid-session table to get broadcast_ids, and save it onto
// the session struct
func (c *Client) sessionBroadcastIDs(ctx context.Context, sess *summary.Session, sessionLockingMap *lockingMap) error {
	tableName := zephyrBroadcastIDSession
	output, err := c.queryZephyrSessionTable(ctx, sess, tableName)
	if err != nil {
		return err
	}

	if *output.Count == 0 {
		return nil
	}

	item := output.Items[0]
	var broadcastIDs []string
	err = dynamodbattribute.Unmarshal(item["broadcast_ids"], &broadcastIDs)
	if err != nil {
		return err
	}

	sess.BroadcastIDs = broadcastIDs
	return nil
}

// sessionVideoPlayGeographics will take a session's SegmentID and the
// query zephyr's video-play-geographical-session table to get broadcast_ids, and
// save it onto the session struct
func (c *Client) sessionVideoPlayGeographics(ctx context.Context, sess *summary.Session, sessionLockingMap *lockingMap) error {
	tableName := zephyrVideoPlayGeographicalSessionTable
	output, err := c.queryZephyrSessionTable(ctx, sess, tableName)
	if err != nil {
		return err
	}

	sessionLockingMap.storeStats("video_play_geographics", nil)

	if *output.Count == 0 {
		return nil
	}

	item := output.Items[0]
	var videoPlayGeographicalMap map[string]int64
	err = dynamodbattribute.Unmarshal(item["country_breakdown"], &videoPlayGeographicalMap)
	if err != nil {
		return errors.Wrap(err, "failed to unmarshal video play country breakdown")
	}

	videoPlayGeographicalArray, totalPlays := convertVideoPlayMapToArray(videoPlayGeographicalMap)
	// we hide demographics data if we're below the legal threshold
	if !legal.DemographicsRevealed(totalPlays) {
		return nil
	}

	topTenVideoPlayGeographical := getTopTenVideoPlay(videoPlayGeographicalArray)
	// LEGAL: we only show the top country if we're below the geo threshold
	// LEGAL: we don't show any countries if we're below the views threshold
	if !(legal.AnyCountryRevealed(totalPlays)) {
		sessionLockingMap.storeStats("video_play_geographics", []*summary.Stat{})
	} else if !(legal.InvididualCountriesRevealed(len(topTenVideoPlayGeographical))) {
		sessionLockingMap.storeStats("video_play_geographics", topTenVideoPlayGeographical[0:1])
	} else {
		sessionLockingMap.storeStats("video_play_geographics", topTenVideoPlayGeographical)
	}

	return nil
}

// sessionVideoPlayPlatforms will take a session's SegmentID and the query
// zephyr's video-play-platform-session table to get broadcast_ids, and save it
// onto the session struct
func (c *Client) sessionVideoPlayPlatforms(ctx context.Context, sess *summary.Session, sessionLockingMap *lockingMap) error {
	tableName := zephyrVideoPlayPlatformSessionTable
	output, err := c.queryZephyrSessionTable(ctx, sess, tableName)
	if err != nil {
		return err
	}

	sessionLockingMap.storeStats("video_play_platforms", nil)

	if *output.Count == 0 {
		return nil
	}

	item := output.Items[0]
	var videoPlayPlatformMap map[string]int64
	err = dynamodbattribute.Unmarshal(item["platform_breakdown"], &videoPlayPlatformMap)
	if err != nil {
		return errors.Wrap(err, "failed to unmarshal video play platform breakdown")
	}

	videoPlayPlatformMap = consolidatePlatforms(videoPlayPlatformMap)
	videoPlayPlatformArray, totalPlays := convertVideoPlayMapToArray(videoPlayPlatformMap)

	if !legal.DemographicsRevealed(totalPlays) {
		return nil
	}

	// LEGAL: show aggregated platform info if we're below the platform threshold
	if !legal.IndividualPlatformRevealed(totalPlays) {
		var aggregate int64

		reducedVideoPlayPlatforms := make([]*summary.Stat, 0, len(videoPlayPlatformArray)+1)

		// aggregate android, ios, and mobile web together
		for _, record := range videoPlayPlatformArray {
			switch record.Key {
			case platformAndroid, platformIOS, platformMobileWeb:
				value, err := strconv.ParseInt(record.Value, 10, 64)
				if err != nil {
					log.WithError(err).Warn("failed to parse video play platform value")
					return nil
				}
				aggregate += value
			default:
				reducedVideoPlayPlatforms = append(reducedVideoPlayPlatforms, record)
			}
		}

		videoPlayPlatformArray = append(reducedVideoPlayPlatforms, &summary.Stat{
			Key:   platformAggregate,
			Value: strconv.FormatInt(aggregate, 10),
		})
	}

	sessionLockingMap.storeStats("video_play_platforms", getTopTenVideoPlay(videoPlayPlatformArray))

	return nil
}

// sessionVideoPlayReferrers will take a session's SegmentID and the query
// zephyr's video-play-platform-session table to get broadcast_ids, and save it
// onto the session struct
func (c *Client) sessionVideoPlayReferrers(ctx context.Context, sess *summary.Session, sessionLockingMap *lockingMap, channelLogin func() (string, error)) error {
	tableName := zephyrVideoPlayReferrerSessionTable
	output, err := c.queryZephyrSessionTable(ctx, sess, tableName)
	if err != nil {
		return err
	}

	sessionLockingMap.storeStats("video_play_external_referrers", nil)
	sessionLockingMap.storeStats("video_play_internal_referrers", nil)

	sessionLockingMap.storeStats("video_play_internal_channels_referrers", nil)
	sessionLockingMap.storeStats("video_play_internal_twitch_referrers", nil)

	if *output.Count == 0 {
		return nil
	}

	item := output.Items[0]
	var internalReferrersMap map[string]int64
	err = dynamodbattribute.Unmarshal(item["internal_referrers"], &internalReferrersMap)
	if err != nil {
		return errors.Wrap(err, "failed to unmarshal internal video play referrers")
	}

	var externalReferrersMap map[string]int64
	err = dynamodbattribute.Unmarshal(item["external_referrers"], &externalReferrersMap)
	if err != nil {
		return errors.Wrap(err, "failed to unmarshal external video play referrers")
	}

	// aggregate various countries' external referrers into buckets. for example
	// google.ca, google.com, google.se should be just google
	for key, value := range externalReferrersMap {
		for _, bucketedExternalReferrer := range bucketedExternalReferrers {
			if strings.Contains(key, bucketedExternalReferrer) {
				externalReferrersMap[bucketedExternalReferrer] += value

				delete(externalReferrersMap, key)
			}
		}
	}

	// look up the userID and then combine referrers with the userID's login
	// and top_nav_bar into to login
	login, err := channelLogin()
	if err != nil {
		log.WithError(err).Warn("failed to get user")
	} else {
		for key, value := range internalReferrersMap {
			if key == login {
				internalReferrersMap["top_nav_bar"] += value
				delete(internalReferrersMap, key)
				break
			}
		}
	}

	internalReferrersArray, totalInternalPlays := convertVideoPlayMapToArray(internalReferrersMap)
	externalReferrersArray, totalExternalPlays := convertVideoPlayMapToArray(externalReferrersMap)

	filteredInternalTwitchReferrers := getInternalTwitchReferrers(internalReferrersArray)
	filteredInternalChannelsReferrers := getInternalChannelReferrers(internalReferrersArray)

	totalPlays := totalInternalPlays + totalExternalPlays
	if !legal.DemographicsRevealed(totalPlays) {
		return nil
	}

	// LEGAL: categorize total number of internal and external as external when
	// we're below the individual referral threshold
	if !legal.IndividualReferralRevealed(totalPlays) {
		individualReferrals := []*summary.Stat{
			{
				Key:   referrerInternalAggregate,
				Value: strconv.FormatInt(totalInternalPlays, 10),
			},
			{
				Key:   referrerExternalAggregate,
				Value: strconv.FormatInt(totalExternalPlays, 10),
			},
		}

		sessionLockingMap.storeStats("video_play_external_referrers", getTopTenVideoPlay(individualReferrals))
		return nil
	}

	// LEGAL: we categorize all individual channel pages as other channel page
	// if we're below channel name reveal threshold
	if !legal.ChannelNameRevealed(totalPlays) {
		var channelAggregateCount int64
		whitelistedInternalArray := make([]*summary.Stat, 0, len(internalReferrersArray))
		for _, record := range internalReferrersArray {
			if _, ok := internalReferrerWhitelist[record.Key]; ok {
				whitelistedInternalArray = append(whitelistedInternalArray, record)
				continue
			}

			value, err := strconv.ParseInt(record.Value, 10, 64)
			if err != nil {
				log.WithError(err).Warn("failed to parse internal referrer value")
				continue
			}

			channelAggregateCount += value
		}

		if channelAggregateCount > 0 {
			whitelistedInternalArray = append(whitelistedInternalArray, &summary.Stat{
				Key:   referrerInternalOtherChannelPage,
				Value: strconv.FormatInt(channelAggregateCount, 10),
			})
		}

		internalReferrersArray = whitelistedInternalArray
	}

	sessionLockingMap.storeStats("video_play_external_referrers", getTopTenVideoPlay(externalReferrersArray))
	sessionLockingMap.storeStats("video_play_internal_referrers", getTopTenVideoPlay(internalReferrersArray))

	sessionLockingMap.storeStats("video_play_internal_twitch_referrers", getTopTenVideoPlay(filteredInternalTwitchReferrers))
	sessionLockingMap.storeStats("video_play_internal_channels_referrers", getTopTenVideoPlay(filteredInternalChannelsReferrers))

	return nil
}

// sessionVideoPlayCounts will take a session's SegmentID and the query
// zephyr's raid-execute-session table to get number_of_raids, and save it onto
// the session struct
func (c *Client) sessionVideoPlayCounts(ctx context.Context, sess *summary.Session, sessionLockingMap *lockingMap) error {
	tableName := zephyrVideoPlayUniqueSessionTable
	output, err := c.queryZephyrSessionTable(ctx, sess, tableName)
	if err != nil {
		return err
	}

	sessionLockingMap.storeSummary("video_play_total", nil)
	sessionLockingMap.storeSummary("video_play_unique", nil)

	if *output.Count == 0 {
		return nil
	}

	item := output.Items[0]
	if _, ok := item["total_views"]; ok {
		sessionLockingMap.storeSummary("video_play_total", item["total_views"].N)
	}
	if _, ok := item["unique_views"]; ok {
		sessionLockingMap.storeSummary("video_play_unique", item["unique_views"].N)
	}

	return nil
}

// sessionSubscriptions will take a session's SegmentID and the query
// zephyr's raid-execute-session table to get number_of_raids, and save it onto
// the session struct
func (c *Client) sessionSubscriptions(ctx context.Context, sess *summary.Session, sessionLockingMap *lockingMap) error {
	tableName := zephyrSubscriptionsPurchaseSuccessSessionTable
	output, err := c.queryZephyrSessionTable(ctx, sess, tableName)
	if err != nil {
		return err
	}

	sessionLockingMap.storeSummary("subscriptions_new", nil)

	if *output.Count == 0 {
		return nil
	}

	item := output.Items[0]
	if _, ok := item["new_subs"]; ok {
		sessionLockingMap.storeSummary("subscriptions_new", item["new_subs"].N)
	}

	return nil
}

func (c *Client) sessionClipCreates(ctx context.Context, sess *summary.Session, sessionLockingMap *lockingMap) error {
	tableName := zephyrVideoPlayClipCreatesSessionTable
	output, err := c.queryZephyrSessionTable(ctx, sess, tableName)
	if err != nil {
		return err
	}

	sessionLockingMap.storeSummary("clip_creates_total", nil)

	if *output.Count == 0 {
		return nil
	}

	item := output.Items[0]
	if _, ok := item["creates"]; ok {
		sessionLockingMap.storeSummary("clip_creates_total", item["creates"].N)
	}

	return nil
}

func (c *Client) sessionClipPlays(ctx context.Context, sess *summary.Session, sessionLockingMap *lockingMap) error {
	tableName := zephyrVideoPlayClipPlaysSessionTable
	output, err := c.queryZephyrSessionTable(ctx, sess, tableName)
	if err != nil {
		return err
	}

	sessionLockingMap.storeSummary("clip_plays_total", nil)
	sessionLockingMap.storeStats("clip_plays_breakdown", nil)

	if *output.Count == 0 {
		return nil
	}

	item := output.Items[0]
	if _, ok := item["plays"]; ok {
		sessionLockingMap.storeSummary("clip_plays_total", item["plays"].N)
	}

	if _, ok := item["referrers"]; ok {
		var playBreakdown map[string]int64
		err := dynamodbattribute.Unmarshal(item["referrers"], &playBreakdown)
		if err != nil {
			return errors.Wrap(err, "failed to unmarshal clip play referrers")
		}

		playBreakdownArray, _ := convertVideoPlayMapToArray(playBreakdown)

		sessionLockingMap.storeStats("clip_plays_breakdown", getTopTenVideoPlay(playBreakdownArray))
	}

	return nil
}

// sessionHostRaidPercentage is a helper function to query a target table and
// and return a pointer of the dynamodb query output
func (c *Client) sessionHostRaidPercentage(ctx context.Context, sess *summary.Session, sessionLockingMap *lockingMap) error {
	tableName := zephyrVideoPlayHostRaidsSessionTable
	output, err := c.queryZephyrSessionTable(ctx, sess, tableName)
	if err != nil {
		return err
	}

	sessionLockingMap.storeSummary("hosted_raided_pct", nil)

	if *output.Count == 0 {
		return nil
	}

	item := output.Items[0]
	if _, ok := item["hosted_raided_pct"]; ok {
		sessionLockingMap.storeSummary("hosted_raided_pct", item["hosted_raided_pct"].N)
	}

	return nil
}

// convertVideoPlayMapToArray expects a video play demographics map to be passed
// in and returns an unsorted array of structs containing session.VideoPlayStat
func convertVideoPlayMapToArray(input map[string]int64) ([]*summary.Stat, int64) {
	output := make([]*summary.Stat, 0, len(input))
	var total int64

	for key, value := range input {
		output = append(output, &summary.Stat{
			Key:   key,
			Value: strconv.FormatInt(value, 10),
		})

		total += value
	}

	return output, total
}

// getTopTenVideoPlay will sort the input array of video play records, and
// return an array of the top ten records
func getTopTenVideoPlay(input []*summary.Stat) []*summary.Stat {
	if len(input) < 2 {
		return input
	}

	sort.Slice(input, func(i, j int) bool {
		itemI, err := strconv.ParseInt(input[i].Value, 10, 64)
		if err != nil {
			log.Error("GetTopTenVideoPlay() failed to parse int", err)
		}

		itemJ, err := strconv.ParseInt(input[j].Value, 10, 64)
		if err != nil {
			log.Error("GetTopTenVideoPlay() failed to parse int", err)
		}

		return itemI > itemJ
	})

	if len(input) < 10 {
		return input
	}

	return input[0:10]
}

func getInternalTwitchReferrers(internals []*summary.Stat) []*summary.Stat {
	filteredTwitchReferrersArray := make([]*summary.Stat, 0, len(internals))
	var channelAggregateCount int64
	for _, record := range internals {
		if _, ok := internalReferrerWhitelist[record.Key]; ok {
			filteredTwitchReferrersArray = append(filteredTwitchReferrersArray, record)
			continue
		}

		value, err := strconv.ParseInt(record.Value, 10, 64)
		if err != nil {
			log.WithError(err).Warn("failed to parse internal referrer value")
			continue
		}

		channelAggregateCount += value
	}

	return filteredTwitchReferrersArray
}

func getInternalChannelReferrers(internals []*summary.Stat) []*summary.Stat {
	filteredChannelReferrersArray := make([]*summary.Stat, 0, len(internals))
	var channelAggregateCount int64
	for _, record := range internals {
		if _, ok := internalReferrerWhitelist[record.Key]; !ok {
			filteredChannelReferrersArray = append(filteredChannelReferrersArray, record)
			continue
		}

		value, err := strconv.ParseInt(record.Value, 10, 64)
		if err != nil {
			log.WithError(err).Warn("failed to parse internal referrer value")
			continue
		}

		channelAggregateCount += value
	}

	return filteredChannelReferrersArray
}
