package zephyr

import (
	"context"
	"strconv"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/pkg/errors"
)

// SubscriptionPurchase is a single timestemp struct
// for a unique subscription success event from zephyr
type SubscriptionPurchase struct {
	ChannelID int64
	Count     int64
	Time      time.Time
}

// GetSubscriptionPurchaseSuccessByTime returns all subscription purchase by time series for
// a given channelID between startTime and endTime. Returns an array of
// SubscriptionPurchase structs or an error if something went wrong. Assumes startTime
// and endTime are valid.
func (c *Client) GetSubscriptionPurchaseSuccessByTime(ctx context.Context, channelID int64, startTime time.Time, endTime time.Time) ([]SubscriptionPurchase, error) {
	tableName := "SubscriptionsPurchaseSuccessTimeseries"

	var exclusiveStartKey map[string]*dynamodb.AttributeValue
	keyCondition := aws.String("channel_id = :channelID AND #T BETWEEN :startTime AND :endTime")
	attributeValues := map[string]*dynamodb.AttributeValue{
		":channelID": {
			S: aws.String(strconv.FormatInt(channelID, 10)),
		},
		":startTime": {
			S: aws.String(startTime.Format(dynamoDBTimeFormat)),
		},
		":endTime": {
			S: aws.String(endTime.Format(dynamoDBTimeFormat)),
		},
	}
	attributeNames := map[string]*string{
		"#T": aws.String("timestamp"),
	}

	subsriptionPurchases := []SubscriptionPurchase{}
	for {
		output, err := c.dynamo.QueryWithContext(ctx, &dynamodb.QueryInput{
			TableName:                 aws.String(tableName),
			ScanIndexForward:          aws.Bool(true),
			KeyConditionExpression:    keyCondition,
			ExpressionAttributeValues: attributeValues,
			ExpressionAttributeNames:  attributeNames,
			ExclusiveStartKey:         exclusiveStartKey,
		})

		if err != nil {
			return nil, errors.Wrapf(err, "failed to query %s", tableName)
		}

		for _, item := range output.Items {
			subscriptionPurchase, err := c.buildSubscriptionPurchaseSuccess(item)
			if err != nil {
				return nil, errors.Wrap(err, "failed to build subscription purchase success")
			}

			subsriptionPurchases = append(subsriptionPurchases, *subscriptionPurchase)
		}

		if output.LastEvaluatedKey == nil {
			break
		}

		exclusiveStartKey = output.LastEvaluatedKey
	}

	return subsriptionPurchases, nil
}

func (c *Client) buildSubscriptionPurchaseSuccess(item map[string]*dynamodb.AttributeValue) (*SubscriptionPurchase, error) {
	channelID, err := strconv.ParseInt(*item["channel_id"].S, 10, 64)
	if err != nil {
		return nil, err
	}

	timetime, err := time.Parse(dynamoDBTimeFormat, *item["timestamp"].S)
	if err != nil {
		return nil, err
	}

	var count int64
	if item["new_subs"] != nil {
		if parsed, err := strconv.ParseInt(*item["new_subs"].N, 10, 64); err == nil {
			count = parsed
		}
	}

	return &SubscriptionPurchase{
		ChannelID: channelID,
		Count:     count,
		Time:      timetime,
	}, nil
}
