package zephyr

import (
	"context"
	"strconv"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/pkg/errors"
)

// UniqueView is a single timestemp struct for a video-play (unique devices) event from zephyr
type UniqueView struct {
	ChannelID   int64
	TotalViews  int64
	UniqueViews int64
	Time        time.Time
}

// GetViewsByTime returns all new views unique and non-unique by time series for
// a given channelID between startTime and endTime. Returns an array of
// UniqueViews structs or an error if something went wrong. Assumes startTime
// and endTime are valid.
func (c *Client) GetViewsByTime(ctx context.Context, channelID int64, startTime time.Time, endTime time.Time) ([]UniqueView, error) {
	tableName := "VideoPlayUniqueTimeseries"

	var exclusiveStartKey map[string]*dynamodb.AttributeValue
	keyCondition := aws.String("channel_id = :channelID AND #T BETWEEN :startTime AND :endTime")
	attributeValues := map[string]*dynamodb.AttributeValue{
		":channelID": {
			S: aws.String(strconv.FormatInt(channelID, 10)),
		},
		":startTime": {
			S: aws.String(startTime.Format(dynamoDBTimeFormat)),
		},
		":endTime": {
			S: aws.String(endTime.Format(dynamoDBTimeFormat)),
		},
	}
	attributeNames := map[string]*string{
		"#T": aws.String("timestamp"),
	}

	uniqueViews := []UniqueView{}
	for {
		output, err := c.dynamo.QueryWithContext(ctx, &dynamodb.QueryInput{
			TableName:                 aws.String(tableName),
			ScanIndexForward:          aws.Bool(true),
			KeyConditionExpression:    keyCondition,
			ExpressionAttributeValues: attributeValues,
			ExpressionAttributeNames:  attributeNames,
			ExclusiveStartKey:         exclusiveStartKey,
		})

		if err != nil {
			return nil, errors.Wrapf(err, "failed to query %s", tableName)
		}

		for _, item := range output.Items {
			uniqueView, err := c.buildUniqueView(item)
			if err != nil {
				return nil, errors.Wrap(err, "failed to build unique view")
			}

			uniqueViews = append(uniqueViews, *uniqueView)
		}

		if output.LastEvaluatedKey == nil {
			break
		}

		exclusiveStartKey = output.LastEvaluatedKey
	}

	return uniqueViews, nil
}

func (c *Client) buildUniqueView(item map[string]*dynamodb.AttributeValue) (*UniqueView, error) {
	channelID, err := strconv.ParseInt(*item["channel_id"].S, 10, 64)
	if err != nil {
		return nil, err
	}

	timetime, err := time.Parse(dynamoDBTimeFormat, *item["timestamp"].S)
	if err != nil {
		return nil, err
	}

	var totalViews, uniqueViews int64
	if item["total_views"] != nil {
		if parsed, err := strconv.ParseInt(*item["total_views"].N, 10, 64); err == nil {
			totalViews = parsed
		}
	}

	if item["unique_views"] != nil {
		if parsed, err := strconv.ParseInt(*item["unique_views"].N, 10, 64); err == nil {
			uniqueViews = parsed
		}
	}

	return &UniqueView{
		ChannelID:   channelID,
		UniqueViews: uniqueViews,
		TotalViews:  totalViews,
		Time:        timetime,
	}, nil
}
